<?php
/**
 * NOTICE OF LICENSE.
 *
 * @file Sitemap generator for SimpleSitemap
 *
 * This source file is subject to a commercial license from Agence Malttt SAS
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the Agence Malttt SAS is strictly forbidden.
 * INFORMATION SUR LA LICENCE D'UTILISATION
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe Agence Malttt SAS
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part d'Agence Malttt SAS est expressement interdite.
 *
 * @author    Matthieu Deroubaix
 * @copyright Copyright (c) 2015-2016 Agence Malttt SAS - 90 Rue faubourg saint martin - 75010 Paris
 * @license   Commercial license
 * Support by mail  :  support@agence-malttt.fr
 * Phone : +33.972535133
 */

class SimpleSitemapSitemapModuleFrontController extends ModuleFrontController
{
    public function initContent()
    {
        parent::initContent();

        require_once _PS_MODULE_DIR_.'simplesitemap/simplesitemap.php';

        $this->mod = new SimpleSitemap();
        $this->context = Context::getContext();
        $this->db = Db::getInstance(_PS_USE_SQL_SLAVE_);
        $this->context_id_shop = (int) Shop::getContextShopID();

        if (isset($_REQUEST['cron']) && $_REQUEST['cron'] == $this->mod->secure_key) {
            try {
                $this->mod->prepareRobots();
            } catch (Exception $e) {
                PrestaShopLogger::addLog('Unable to modify robots.txt');
            }

            $domains = ShopUrl::getShopUrls();

            if (!empty($domains)) {
                $search_engines_pings = array(
                    'http://blogs.yandex.ru/pings/?status=success&url=',
                    'http://www.bing.com/webmaster/ping.aspx?siteMap=',
                    'http://www.google.com/webmasters/tools/ping?sitemap=',
                );

                $all_langs = Configuration::get('SIMPSMP_LANG_MODE', null, null, (int) $this->context_id_shop);

                foreach ($domains as $shop_url) {
                    if ((int) $shop_url->active != 1) {
                        continue;
                    }

                    $langs = $this->mod->getSitemapLanguages($shop_url->id, $all_langs);

                    foreach ($search_engines_pings as $url) {
                        foreach ($langs as $lang) {
                            $sitemap_url = $this->mod->getSitemapLink('index', $lang['id_lang'], $shop_url->id);
                            Tools::file_get_contents($url.urlencode($sitemap_url));
                        }
                    }
                }
            }

            die;
        }

        header('Content-Type: text/xml; charset=utf-8');

        $sitemap_type = Tools::getValue('rewrite');

        if (!empty($sitemap_type)) {
            $sitemap_type = str_replace('_', '', $sitemap_type);
        }

        $sitemaps_available = array(
            'index',
            'manufacturers',
            'ph-simpleblog-posts',
            'ph-simpleblog-categories',
            'smartblog-posts',
            'smartblog-categories',
            'advanceblog-posts',
            'advanceblog-categories',
            'amazzingblog-posts',
            'amazzingblog-categories',
            'stblog-posts',
            'stblog-categories',
            'leoblog-posts',
            'leoblog-categories',
            'categories',
            'others',
            'cms',
            'suppliers',
        );

        $product_pages = $this->getProductsPages($this->context_id_shop);

        for ($i = 1; $i <= $product_pages; ++$i) {
            $sitemaps_available[] = 'products-'.$i;
        }
        
        $this->fallback_lang = (bool) Configuration::get('SIMPSMP_LANG_MODE') == true ? Context::getContext()->language->id : null;

        if (!in_array($sitemap_type, $sitemaps_available)) {
            $url = $this->mod->getSitemapLink('index', $this->fallback_lang, $this->context_id_shop);
            Tools::redirect($url);
        }

        $this->removed_links = $this->getRemovedLinks();

        $this->amazzingblog_index_url = array();

        echo $this->generateSitemap($sitemap_type);

        die;
    }

    private function getRemovedLinks()
    {
        $array = array();

        $sql = $this->db->executeS('SELECT url FROM '._DB_PREFIX_.'simplesitemapremove WHERE 1');

        if (!empty($sql)) {
            foreach ($sql as $value) {
                $array[] = $value['url'];
            }
        }

        return $array;
    }

    private function getProductsPages($id_shop)
    {
        if (!isset($this->nb_productpages[$id_shop])) {
            $result = $this->db->executeS('
                SELECT p.id_product
                FROM '._DB_PREFIX_.'product p
                LEFT JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product = p.id_product AND ps.id_shop = '.(int) $id_shop.')
                WHERE 
                ps.`active` = 1
                AND ps.id_shop = '.(int) $id_shop.'
                GROUP BY p.id_product
                ORDER BY p.id_product DESC
            ');

            $total_products = count($result);
            $this->nb_productpages[$id_shop] = ceil($total_products / 5000);
        }


        return $this->nb_productpages[$id_shop];
    }

    private function generateSitemap($sitemap_type)
    {

        // Setup
        $id_shop = $this->context_id_shop;

        if (empty($id_shop)) {
            $id_shop = 1;
        }

        if ((bool) Configuration::get('SIMPSMP_LANG_MODE') == true) {
            $not_in_lang = Configuration::get('SIMPSMP_EXC_LANG', null, null, (int) $id_shop);
            $exp = array();

            if (!empty($not_in_lang)) {

                // Sanitize
                $exp = explode(',', trim($not_in_lang));
                $exp = array_filter($exp);
            }

            $id_lang_default = (int) Context::getContext()->language->id;

            $iso = Language::getIsoById($id_lang_default);

            if (!in_array($iso, $exp)) {
                $langs = array();
                $langs[0] = array('id_lang' => $id_lang_default);
            } else {
                return;
            }
        } else {
            $sql_not_in = ' ';

            $not_in_lang = Configuration::get('SIMPSMP_EXC_LANG', null, null, (int) $id_shop);

            if (!empty($not_in_lang)) {

                // Sanitize
                $exp = explode(',', trim($not_in_lang));

                $exp = array_filter($exp);

                $exp_array = array();

                foreach ($exp as $val) {
                    $exp_array[] = '"'.trim($val).'"';
                }

                $imp = implode(',', $exp_array);

                $sql_not_in .= ' AND l.iso_code NOT IN ('.$imp.') ';
            }

            $sql = 'SELECT l.id_lang FROM '._DB_PREFIX_.'lang l INNER JOIN '._DB_PREFIX_.'lang_shop sl ON (sl.id_shop = '.(int) $id_shop.' AND sl.id_lang = l.id_lang) WHERE l.active = 1 '.$sql_not_in;

            $langs = Db::getInstance()->executeS($sql);
        }
        
        $this->home_dir = $this->getBaseLink($id_shop);
        $md5 = md5(serialize($langs));
        $cache_file = $this->mod->cache_folder.$id_shop.'_'.$md5.'_'.$sitemap_type.'.xml';
        $now = new \Datetime('now');
        $sitemap = null;
        $priority = 1;
        $count = 0.01;
        $link = new Link();
        $links = array();        // Avoid duplicate

        $freq = Configuration::get('SIMPSMP_FREQ_'.Tools::strtoupper($sitemap_type));

        if (empty($freq)) {
            $freq = Configuration::get('SIMPSMP_FREQ_BLOG');
        }

        // Better checking this before
        if (is_writable($this->mod->cache_folder)) {
            $time_cache = (int) Configuration::get('SIMPSMP_CACHETIME');

            if (file_exists($cache_file) && (filemtime($cache_file) > (time() - ($time_cache * 60)))) {
                return Tools::file_get_contents($cache_file);
            }
        }

        $date = date('Y-m-d');

        if ($sitemap_type == 'index') {
            $xmlString = <<<XML
<?xml version="1.0" encoding="UTF-8" ?>
<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">
</sitemapindex>
XML;
        } else {
            $xmlString = <<<XML
<?xml version="1.0" encoding="UTF-8" ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">
</urlset>
XML;
        }

        $xml = new \SimpleXMLElement($xmlString);

        if (!empty($langs)) {
            if ($sitemap_type == 'index') {
                if (Configuration::get('SIMPSMP_CMS')) {
                    $this->addIndex($xml, $this->mod->getSitemapLink('cms', $this->fallback_lang, $id_shop), $date);
                }

                if (Configuration::get('SIMPSMP_CATEGORIES')) {
                    $this->addIndex($xml, $this->mod->getSitemapLink('categories', $this->fallback_lang, $id_shop), $date);
                }

                if (Configuration::get('SIMPSMP_MANUFACTURERS')) {
                    $this->addIndex($xml, $this->mod->getSitemapLink('manufacturers', $this->fallback_lang, $id_shop), $date);
                }

                if (Configuration::get('SIMPSMP_SUPPLIERS')) {
                    $this->addIndex($xml, $this->mod->getSitemapLink('suppliers', $this->fallback_lang, $id_shop), $date);
                }

                $sql_others = $this->db->executeS('SELECT * FROM '._DB_PREFIX_.'simplesitemapother WHERE 1');

                if (Configuration::get('SIMPSMP_OTHERS') && !empty($sql_others)) {
                    $this->addIndex($xml, $this->mod->getSitemapLink('others', $this->fallback_lang, $id_shop), $date);
                }

                if (Configuration::get('SIMPSMP_BLOG')) {
                    if (Module::isInstalled('ph_simpleblog') && Module::isEnabled('ph_simpleblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('ph-simpleblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('ph-simpleblog-categories', $this->fallback_lang, $id_shop), $date);
                    }

                    if (Module::isInstalled('smartblog') && Module::isEnabled('smartblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('smartblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('smartblog-categories', $this->fallback_lang, $id_shop), $date);
                    }

                    if (Module::isInstalled('advanceblog') && Module::isEnabled('advanceblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('advanceblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('advanceblog-categories', $this->fallback_lang, $id_shop), $date);
                    }
                    
                    if (Module::isInstalled('amazzingblog') && Module::isEnabled('amazzingblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('amazzingblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('amazzingblog-categories', $this->fallback_lang, $id_shop), $date);
                    }

                    if (Module::isInstalled('stblog') && Module::isEnabled('stblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('stblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('stblog-categories', $this->fallback_lang, $id_shop), $date);
                    }
                    
                    if (Module::isInstalled('leoblog') && Module::isEnabled('leoblog')) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('leoblog-posts', $this->fallback_lang, $id_shop), $date);
                        $this->addIndex($xml, $this->mod->getSitemapLink('leoblog-categories', $this->fallback_lang, $id_shop), $date);
                    }
                    
                    if (Module::isInstalled('blockblog') && Module::isEnabled('blockblog')) {
                        if (defined('_PS_HOST_MODE_')) {
                            $blog_link = ''._PS_BASE_URL_.__PS_BASE_URI__.'modules/blockblog/upload/blog.xml';
                        } else {
                            // Cloud mode
                            $blog_link .= ''._PS_BASE_URL_.__PS_BASE_URI__.'upload/blockblog/blog.xml';
                        }

                        $this->addIndex($xml, $blog_link, $date);
                    }

                    if (Module::isInstalled('PrestaBlog') && Module::isEnabled('PrestaBlog')) {
                        $blog_link = _PS_MODULE_DIR_.'/prestablog/sitemap/'.$id_shop.'/master.xml';

                        if (file_exists($blog_link) && is_readable($blog_link)) {
                            $string = Tools::file_get_contents($blog_link);
                            $regex = '/\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|$!:,.;]*[A-Z0-9+&@#\/%=~_|$]/i';
                            preg_match_all($regex, $string, $matches);

                            if (isset($matches[0]) && !empty($matches[0])) {
                                $urls = $matches[0];

                                foreach ($urls as $url) {
                                    if (strpos($url, $this->getBaseLink()) !== false) {
                                        $this->addIndex($xml, $url, $date);
                                    }
                                }
                            }
                        }
                    }
                }

                if (Configuration::get('SIMPSMP_PRODUCTS')) {
                    $nb_pages = $this->getProductsPages($id_shop);
                    for ($i = 1; $i <= $nb_pages; ++$i) {
                        $this->addIndex($xml, $this->mod->getSitemapLink('products-'.$i, $this->fallback_lang, $id_shop), $date);
                    }
                }
            } elseif ($sitemap_type == 'cms') {
                $this->addRow($xml, $this->home_dir, $priority, $freq, $date);

                /* CMS Generator */
                if (Configuration::get('SIMPSMP_CMS') || !Module::isInstalled('blockcms')) {
                    $sql_cms = '
                    SELECT DISTINCT '.(Configuration::get('PS_REWRITING_SETTINGS') ? 'cl.id_cms, cl.link_rewrite, cl.id_lang' : 'cl.id_cms').
                    ' FROM '._DB_PREFIX_.'cms_lang cl
                    LEFT JOIN '._DB_PREFIX_.'lang l ON (cl.id_lang = l.id_lang)
                    WHERE l.`active` = 1
                    GROUP BY cl.id_cms
                    ORDER BY cl.id_cms, cl.id_lang ASC ';
                } elseif (Module::isInstalled('blockcms')) {
                    $sql_cms = '
                    SELECT DISTINCT '.(Configuration::get('PS_REWRITING_SETTINGS') ? 'cl.id_cms, cl.link_rewrite, cl.id_lang' : 'cl.id_cms').
                    ' FROM '._DB_PREFIX_.'cms_block_page b
                    LEFT JOIN '._DB_PREFIX_.'cms_lang cl ON (b.id_cms = cl.id_cms)
                    LEFT JOIN '._DB_PREFIX_.'lang l ON (cl.id_lang = l.id_lang)
                    WHERE l.`active` = 1 GROUP BY cl.id_cms
                    ORDER BY cl.id_cms, cl.id_lang ASC';
                }

                $cmss = $this->db->executeS($sql_cms);

                foreach ($langs as $lang) {
                    foreach ($cmss as $cms) {
                        $id_lang = $lang['id_lang'];
                        $tmpLink = $link->getCMSLink((int) $cms['id_cms'], null, null, $id_lang);

                        if (!in_array($tmpLink, $links)) {
                            $links[] = $tmpLink;
                            $this->addRow($xml, $tmpLink, '0.8', $freq);
                        }
                    }
                }

                // Meta only if url rewriting is ON
                if ((bool) Configuration::get('PS_REWRITING_SETTINGS') == true) {
                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        $context = Context::getContext();
                        $context->language->id = $id_lang;
                        $metas = Meta::getMetasByIdLang((int) $id_lang);

                        $url = $this->getBaseLink($id_shop).$this->getLangLink($id_lang, $id_shop);

                        if (count($metas) > 0) {
                            $downgrade = (1 / count($metas)) / 10;
                            foreach ($metas as $value) {
                                $tmpLink = $url.$value['url_rewrite'];
                                if (!in_array($tmpLink, $links)) {
                                    $links[] = $tmpLink;
                                    $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, $now->format('Y-m-d'));
                                    $priority = $priority - $downgrade;
                                }
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'others') {
                $conf = Configuration::get('SIMPSMP_OTHERS');

                if (!empty($conf)) {
                    $sql_others = $this->db->executeS('SELECT * FROM '._DB_PREFIX_.'simplesitemapother WHERE 1');

                    if (!empty($sql_others)) {
                        foreach ($sql_others as $value) {
                            if (!in_array($value['url'], $links)) {
                                $downgrade = (1 / count($sql_others)) / 10;
                                $links[] = $value['url'];
                                $sitemap = $this->addRow($xml, $value['url'], $priority, $freq, $now->format('Y-m-d'));
                                $priority = $priority - $downgrade;
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'manufacturers') {
                $conf = Configuration::get('SIMPSMP_MANUFACTURERS');

                if (!empty($conf)) {

                    /* Manufacturers Generator */
                    $sql = 'SELECT c.id_manufacturer, c.name, cl.id_lang, DATE_FORMAT(IF(date_upd,date_upd,date_add), \'%Y-%m-%d\') AS date_upd
                    FROM '._DB_PREFIX_.'manufacturer c
                    LEFT JOIN '._DB_PREFIX_.'manufacturer_shop cs ON c.id_manufacturer = cs.id_manufacturer
                    LEFT JOIN '._DB_PREFIX_.'manufacturer_lang cl ON c.id_manufacturer = cl.id_manufacturer
                    LEFT JOIN '._DB_PREFIX_.'lang l ON (cl.id_lang = l.id_lang)
                    WHERE c.active = 1 AND l.`active` = 1 AND cs.id_shop='.(int) $id_shop.' 
                    GROUP BY c.id_manufacturer
                    ORDER BY c.id_manufacturer DESC';

                    $manufacturers = $this->db->executeS($sql);

                    if (count($manufacturers) > 0) {
                        foreach ($langs as $lang) {
                            $downgrade = (1 / count($manufacturers)) / 10;

                            foreach ($manufacturers as $manufacturer) {
                                $tmpLink = $link->getManufacturerLink((int) $manufacturer['id_manufacturer'], null, (int) $lang['id_lang'], $id_shop);

                                if (!in_array($tmpLink, $links)) {
                                    $links[] = $tmpLink;
                                    $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($manufacturer['date_upd'], 0, 10));
                                    $priority = $priority - $downgrade;
                                }
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'suppliers') {
                $conf = Configuration::get('SIMPSMP_SUPPLIERS');

                if (!empty($conf)) {
                    $sql = 'SELECT c.id_supplier, c.name, cl.id_lang, DATE_FORMAT(IF(date_upd,date_upd,date_add), \'%Y-%m-%d\') AS date_upd
                    FROM '._DB_PREFIX_.'supplier c
                    LEFT JOIN '._DB_PREFIX_.'supplier_shop cs ON c.id_supplier = cs.id_supplier
                    LEFT JOIN '._DB_PREFIX_.'supplier_lang cl ON c.id_supplier = cl.id_supplier
                    LEFT JOIN '._DB_PREFIX_.'lang l ON (cl.id_lang = l.id_lang)
                    WHERE c.active = 1 AND l.`active` = 1 AND cs.id_shop='.(int) $id_shop.' 
                    GROUP BY c.id_supplier
                    ORDER BY c.id_supplier DESC';

                    $suppliers = $this->db->executeS($sql);

                    if (count($suppliers) > 0) {
                        foreach ($langs as $lang) {
                            $downgrade = (1 / count($suppliers)) / 10;

                            foreach ($suppliers as $supplier) {
                                $tmpLink = $link->getSupplierLink((int) $supplier['id_supplier'], null, (int) $lang['id_lang'], $id_shop);

                                if (!in_array($tmpLink, $links)) {
                                    $links[] = $tmpLink;
                                    $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($supplier['date_upd'], 0, 10));
                                    $priority = $priority - $downgrade;
                                }
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'categories') {
                $conf = Configuration::get('SIMPSMP_CATEGORIES');

                if (!empty($conf)) {
                    $img_type = Configuration::get('SIMPSMP_C_IMG');

                    $l = array();

                    foreach ($langs as $value) {
                        $l[] = $value['id_lang'];
                    }

                    /* Categories Generator */
                    $categories = $this->db->executeS('SELECT c.id_category, c.level_depth, cl.name, cl.link_rewrite, cl.id_lang, DATE_FORMAT(IF(date_upd,date_upd,date_add), \'%Y-%m-%d\') AS date_upd
                        FROM '._DB_PREFIX_.'category c
                        LEFT JOIN '._DB_PREFIX_.'category_shop cs ON c.id_category = cs.id_category
                        LEFT JOIN '._DB_PREFIX_.'category_lang cl ON (c.id_category = cl.id_category AND cl.id_lang IN ('.implode(',', $l).'))
                        LEFT JOIN '._DB_PREFIX_.'lang l ON (cl.id_lang = l.id_lang)
                        INNER JOIN '._DB_PREFIX_.'lang_shop sl ON (sl.id_shop = '.(int) $id_shop.')
                        WHERE c.active = 1 AND l.`active` = 1 AND cs.id_shop='.(int) $id_shop.' AND cl.id_shop='.(int) $id_shop.' AND c.id_category > 2
                        GROUP BY c.id_category
                        ORDER BY c.id_category DESC');

                    foreach ($langs as $lang) {
                        $downgrade = (1 / count($categories)) / 10;
                        foreach ($categories as $category) {
                            if (($priority = 0.9 - ($category['level_depth'] / 10)) < 0.1) {
                                $priority = 0.1;
                            }

                            $tmpLink = $link->getCategoryLink((int) $category['id_category'], null, (int) $lang['id_lang']);
                            if (!in_array($tmpLink, $links)) {
                                $links[] = $tmpLink;
                                $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($category['date_upd'], 0, 10));
                            }

                            $tmpLink = $this->getCatImageLink($category['link_rewrite'], (int) $category['id_category'], $img_type);
                            if (!empty($tmpLink) && !in_array($tmpLink, $links)) {
                                $links[] = $tmpLink;
                                $this->addRowImage($sitemap, $tmpLink, $category['name']);
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'ph-simpleblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('ph_simpleblog') && Module::isEnabled('ph_simpleblog')) {
                    require_once _PS_MODULE_DIR_.'ph_simpleblog/ph_simpleblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;

                        $priority = 1;

                        $posts = SimpleBlogPost::getPosts($id_lang, 5000);
                        $count = 0.05;

                        foreach ($posts as $value) {
                            $tmpLink = SimpleBlogPost::getLink($value['link_rewrite'], $value['category_rewrite']);
                            if (!in_array($tmpLink, $links)) {
                                $priority = $priority - $count;
                                $links[] = $tmpLink;
                                $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($value['date_upd'], 0, 10));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'ph-simpleblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('ph_simpleblog') && Module::isEnabled('ph_simpleblog')) {
                    require_once _PS_MODULE_DIR_.'ph_simpleblog/ph_simpleblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = ph_simpleblog::getLink();

                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));

                        $categories = SimpleBlogCategory::getCategories($id_lang);
                        $count = 0.05;

                        foreach ($categories as $value) {
                            if (!in_array($value['url'], $links)) {
                                $priority = $priority - $count;
                                $links[] = $value['url'];
                                $sitemap = $this->addRow($xml, $value['url'], $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'smartblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('smartblog') && Module::isEnabled('smartblog')) {
                    require_once _PS_MODULE_DIR_.'smartblog/smartblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;

                        $priority = 1;

                        $posts = SmartBlogPost::getAllPost($id_lang, 0, 5000);
                        $count = 0.05;

                        foreach ($posts as $value) {
                            $tmpLink = smartblog::GetSmartBlogLink('smartblog_post', array("id_post" => $value["id_post"], "slug" => $value['link_rewrite']));
                            if (!in_array($tmpLink, $links)) {
                                $priority = $priority - $count;
                                $links[] = $tmpLink;
                                $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($value['created'], 0, 10));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'smartblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('smartblog') && Module::isEnabled('smartblog')) {
                    require_once _PS_MODULE_DIR_.'smartblog/smartblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = smartblog::GetSmartBlogUrl();

                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));

                    
                        $BlogCategory = new BlogCategory();

                        $categories = $BlogCategory->getCategory(1, $id_lang);

                        $count = 0.05;

                        foreach ($categories as $value) {
                            $url = smartblog::GetSmartBlogLink('smartblog_category', array("id_category" => $value["id_smart_blog_category"], "slug" => $value['link_rewrite']));
                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'stblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('stblog') && Module::isEnabled('stblog')) {
                    require_once _PS_MODULE_DIR_.'stblog/stblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = $link->getModuleLink('stblog', 'default');

                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));

                        $categories = StBlogCategory::getSimpleCategories($id_lang);

                        $count = 0.05;

                        foreach ($categories as $value) {
                            $category = new StBlogCategory($value['id_st_blog_category']);
                            $url = $link->getModuleLink('stblog', 'category', array('blog_id_category'=>$category->id,'rewrite'=>$category->link_rewrite[$id_lang]));

                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'stblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('stblog') && Module::isEnabled('stblog')) {
                    require_once _PS_MODULE_DIR_.'stblog/stblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');

                        $posts = StBlogClass::getRecentArticles($id_lang, 10000, Context::getContext());
                        $count = 0.05;

                        foreach ($posts as $value) {
                            if (!in_array($value['link'], $links)) {
                                $priority = $priority - $count;
                                $links[] = $value['link'];
                                $sitemap = $this->addRow($xml, $value['link'], $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'leoblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('leoblog') && Module::isEnabled('leoblog')) {
                    require_once _PS_MODULE_DIR_.'leoblog/leoblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = $link->getModuleLink('leoblog', 'list');

                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));

                        $cat = new Leoblogcat();
                        $categories = $cat->getChild(null, $id_lang, $id_shop, true);

                        $count = 0.05;

                        foreach ($categories as $value) {
                            $url = $link->getModuleLink('leoblog', 'category', array('id'=>$value['id_leoblogcat'],'rewrite'=>$value['link_rewrite']));

                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'leoblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('leoblog') && Module::isEnabled('leoblog')) {
                    require_once _PS_MODULE_DIR_.'leoblog/leoblog.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');

                        $posts = LeoBlogBlog::getListBlogs(null, $id_lang, 1, 10000, 'id_leoblog_blog', 'DESC', array(), true);
                        $count = 0.05;

                        foreach ($posts as $value) {
                            $url = $link->getModuleLink('leoblog', 'blog', array('id'=>$value['id_leoblog_blog'],'rewrite'=>$value['link_rewrite']));

                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'amazzingblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('amazzingblog') && Module::isEnabled('amazzingblog')) {
                    require_once _PS_MODULE_DIR_.'amazzingblog/amazzingblog.php';
                    $amazzingblog = new AmazzingBlog;
                    $first_lang_iso = "";
                    $error_root_url = $amazzingblog->getBaseUri();

                    foreach ($langs as $lang) {
                        if (empty($first_lang_iso)) {
                            $first_lang_iso = Language::getIsoById($lang['id_lang']);
                        }

                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $index_url = $this->getAmazzingBlogIndexURI($id_lang, $id_shop, $amazzingblog->root_id);

                        $posts = $this->db->executeS('
                            SELECT * FROM '._DB_PREFIX_.'a_blog_post p
                            INNER JOIN '._DB_PREFIX_.'a_blog_post_lang pl
                                ON pl.id_post = p.id_post AND pl.id_lang = '.(int) $id_lang.' AND pl.id_shop = '.(int) $id_shop.'
                            WHERE p.active = 1');

                        $count = 0.05;

                        foreach ($posts as $value) {
                            $url = $amazzingblog->getPostLink($value['id_post'], $value['link_rewrite']);
                            $url = str_replace("/".$first_lang_iso."/", "/".Language::getIsoById($id_lang)."/", $url);
                            $url = str_replace($error_root_url, $index_url, $url);

                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'amazzingblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('amazzingblog') && Module::isEnabled('amazzingblog')) {
                    require_once _PS_MODULE_DIR_.'amazzingblog/amazzingblog.php';
                    $first_lang_iso = "";
                    $amazzingblog = new AmazzingBlog;
                    $error_root_url = $amazzingblog->getBaseUri();

                    foreach ($langs as $lang) {
                        if (empty($first_lang_iso)) {
                            $first_lang_iso = Language::getIsoById($lang['id_lang']);
                        }

                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = $this->getAmazzingBlogIndexURI($id_lang, $id_shop, $amazzingblog->root_id);
                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');

                        if (!in_array($index_url, $links)) {
                            $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));
                            $links[] = $index_url;
                        }

                        $categories = $this->db->executeS('
                            SELECT * FROM '._DB_PREFIX_.'a_blog_category c
                            INNER JOIN '._DB_PREFIX_.'a_blog_category_lang cl
                                ON cl.id_category = c.id_category AND cl.id_lang = '.(int)$id_lang.' AND cl.id_shop = '.(int)$id_shop.'
                            WHERE c.active = 1 AND c.id_category <> '.(int)$amazzingblog->root_id);

                        $count = 0.05;

                        foreach ($categories as $value) {
                            $url = $amazzingblog->getCategoryLink($value['id_category'], $value['link_rewrite']);
                            $url = str_replace("/".$first_lang_iso."/", "/".Language::getIsoById($id_lang)."/", $url);
                            $url = str_replace($error_root_url, $index_url, $url);

                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'advanceblog-categories') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('advanceblog') && Module::isEnabled('advanceblog')) {
                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $index_url = $link->getModuleLink('advanceblog', 'blog');

                        $priority = 1;
                        $now = new \Datetime('now - 10 minutes');
                        $sitemap = $this->addRow($xml, $index_url, $priority, $freq, $now->format('Y-m-d'));

                        $sql = 'SELECT fmc.* , fmcl.*

                                FROM '._DB_PREFIX_.'fme_blog_categories as fmc

                                LEFT JOIN `'._DB_PREFIX_.'fme_blog_categories_lang` fmcl ON (

                                fmc.`id_blogcategory` = fmcl.`id_blogcategory`

                                AND fmcl.`id_lang` = '.(int) $id_lang.')

                                WHERE fmc.active = 1

                                GROUP BY fmc.id_blogcategory

                                ORDER BY fmc.position';

                        $res = $this->db->executeS($sql);

                        $count = 0.05;

                        foreach ($res as $value) {
                            $url = $link->getModuleLink('advanceblog', 'blogcat', array('id' => $value['id_blogcategory'], 'cat' => $value['link_rewrite_cat']));
                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif ($sitemap_type == 'advanceblog-posts') {
                $conf = Configuration::get('SIMPSMP_BLOG');

                if (!empty($conf) && Module::isInstalled('advanceblog') && Module::isEnabled('advanceblog')) {
                    require_once _PS_MODULE_DIR_.'advanceblog/models/BlogCategories.php';

                    foreach ($langs as $lang) {
                        $id_lang = $lang['id_lang'];
                        Context::getContext()->language->id = $id_lang;
                        $priority = 1;

                        $sql = 'SELECT fmc.id_post, fmcl.link_rewrite

                            FROM '._DB_PREFIX_.'fme_blog_posts as fmc

                            LEFT JOIN `'._DB_PREFIX_.'fme_blog_posts_lang` fmcl ON (

                            fmc.`id_post` = fmcl.`id_post` AND fmcl.`id_lang` = '.(int) $id_lang.')';

                        $res = $this->db->executeS($sql);

                        foreach ($res as $value) {
                            $url = $link->getModuleLink('advanceblog', 'detail', array('id' => $value['id_post'], 'post' => $value['link_rewrite']));
                            if (!in_array($url, $links)) {
                                $priority = $priority - $count;
                                $links[] = $url;
                                $sitemap = $this->addRow($xml, $url, $priority, $freq, $now->format('Y-m-d'));
                            }
                        }
                    }
                }
            } elseif (Tools::substr($sitemap_type, 0, 9) == 'products-') {
                $conf = Configuration::get('SIMPSMP_PRODUCTS');

                if (!empty($conf)) {
                    $page = Tools::substr($sitemap_type, 9);

                    $start = ($page - 1) * 5000;

                    $l = array();

                    foreach ($langs as $value) {
                        $l[] = $value['id_lang'];
                    }

                    $img_type = Configuration::get('SIMPSMP_P_IMG');

                    /* Product Generator */
                    $products = $this->db->executeS('SELECT p.id_product, pl.link_rewrite, pl.name , DATE_FORMAT(IF(ps.date_upd,ps.date_upd,ps.date_add), \'%Y-%m-%d\') date_upd, pl.id_lang, cl.`link_rewrite` category, ean13, (
                                SELECT MIN(level_depth)
                                FROM '._DB_PREFIX_.'product p2
                                '.Shop::addSqlAssociation('product', 'p2').'
                                LEFT JOIN '._DB_PREFIX_.'category_product cp2 ON p2.id_product = cp2.id_product
                                LEFT JOIN '._DB_PREFIX_.'category c2 ON cp2.id_category = c2.id_category
                                WHERE p2.id_product = p.id_product AND p2.`active` = 1 AND c2.`active` = 1) AS level_depth
                            FROM '._DB_PREFIX_.'product p
                            LEFT JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product = p.id_product AND ps.id_shop = '.(int) $id_shop.')
                            LEFT JOIN '._DB_PREFIX_.'product_lang pl ON (p.id_product = pl.id_product AND pl.id_lang IN('.pSQL(implode(',', $l)).') AND pl.id_shop = '.(int) $id_shop.')
                            LEFT JOIN '._DB_PREFIX_.'category_lang cl ON (ps.id_category_default = cl.id_category AND pl.id_lang = cl.id_lang AND cl.id_shop = '.(int) $id_shop.')
                            LEFT JOIN '._DB_PREFIX_.'lang l ON (pl.id_lang = l.id_lang)
                            WHERE l.`active` = 1
                                AND ps.`active` = 1
                                AND ps.id_shop = '.(int) $id_shop.'
                            '.(Configuration::get('SIMPSMP_PRODUCTS') ? '' : 'HAVING level_depth IS NOT NULL').'
                            GROUP BY pl.id_product
                            ORDER BY pl.id_product, pl.id_lang ASC
                            LIMIT '.$start.',5000');

                    foreach ($langs as $lang) {
                        foreach ($products as $product) {
                            if (($priority = 0.7 - ($product['level_depth'] / 10)) < 0.1) {
                                $priority = 0.1;
                            }

                            $tmpLink = $link->getProductLink((int) ($product['id_product']), $product['link_rewrite'], $product['category'], $product['ean13'], $lang['id_lang'], $id_shop);

                            if (!in_array($tmpLink, $links)) {
                                $links[] = $tmpLink;
                                $sitemap = $this->addRow($xml, $tmpLink, $priority, $freq, Tools::substr($product['date_upd'], 0, 10));

                                $images = $this->db->executeS('SELECT i.id_image, il.legend legend_image
                                FROM '._DB_PREFIX_.'image i
                                LEFT JOIN '._DB_PREFIX_.'image_lang il ON (i.id_image = il.id_image)
                                WHERE `id_product` = '.(int) ($product['id_product']).' AND il.id_lang = '.(int) ($lang['id_lang']).'
                                ORDER BY `position`');

                                foreach ($images as $image) {
                                    $tmpLink = $this->getImageLink($product['link_rewrite'], (int) $product['id_product'].'-'.(int) $image['id_image'], $img_type);

                                    if (!empty($tmpLink) && !in_array($tmpLink, $links)) {
                                        $links[] = $tmpLink;
                                        if ($image['legend_image'] != '') {
                                            $legend_image = $image['legend_image'];
                                        } else {
                                            $legend_image = $product['name'];
                                        }
                                        $this->addRowImage($sitemap, $tmpLink, $legend_image);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        $xml_string = $xml->asXML();

        if (file_exists($this->mod->cache_folder) && is_writable($this->mod->cache_folder)) {
            try {
                file_put_contents($cache_file, $xml_string, LOCK_EX);
            } catch (Exception $e) {
                PrestaShopLogger::addLog('Sitemap Cache folder not writable, please check your modules folder permissions.');
            }
        } else {
            if (is_writable(_PS_CACHE_DIR_) && !file_exists($this->mod->cache_folder)) {
                @mkdir($this->mod->cache_folder);
                @chmod($this->mod->cache_folder, 777);
            }
        }

        return $xml_string;
    }

    private function getRewrite($table, $id_lang, $id_object, $name, $number = 0)
    {
        if ($number > 0) {
            $link_rewrite = Tools::link_rewrite($name.'-'.(string) $number);
        } else {
            $link_rewrite = Tools::link_rewrite($name);
        }

        $test = $this->db->getRow('SELECT * FROM '._DB_PREFIX_.pSQL($table).'_lang WHERE link_rewrite = "'.pSQL($link_rewrite).'" AND id_'.pSQL($table).' != '.pSQL($id_object).' AND id_lang = '.(int) $id_lang);

        if (!empty($test)) {
            return $this->getRewrite($table, $id_lang, $id_object, $name, $number + 1);
        } else {
            return $link_rewrite;
        }
    }

    private function getBaseLink($id_shop = null, $ssl = null, $relative_protocol = false)
    {
        // Old bridge
        return $this->mod->getBaseLink($id_shop, $ssl, $relative_protocol);
    }

    private function getLangLink($id_lang = null, $id_shop = null)
    {
        $allow = (int) Configuration::get('PS_REWRITING_SETTINGS');

        if ((!$allow && in_array($id_shop, array($this->context->shop->id,  null))) || !Language::isMultiLanguageActivated($id_shop) || !(int) Configuration::get('PS_REWRITING_SETTINGS', null, null, $id_shop)) {
            return '';
        }

        if (!$id_lang) {
            $id_lang = $this->context->language->id;
        }

        return Language::getIsoById($id_lang).'/';
    }

    private function getCatImageLink($name, $id_category, $type = null)
    {
        if (!file_exists(_PS_IMG_DIR_.'c/'.$id_category.'-'.$type.'/'.$name.'.jpg')) {
            return false;
        }
        if ((int) Configuration::get('PS_REWRITING_SETTINGS') == 1 && $type) {
            $uri_path = 'c/'.$id_category.'-'.$type.'/'.$name.'.jpg';
        } else {
            $uri_path = _THEME_CAT_DIR_.$id_category.($type ? '-'.$type : '').'.jpg';
        }

        return $this->getBaseLink().ltrim($uri_path, '/');
    }

    private function getImageLink($name, $ids, $type = null)
    {
        $not_default = false;

        // Check if module is installed, enabled, customer is logged in and watermark logged option is on
        if (Configuration::get('WATERMARK_LOGGED') && (Module::isInstalled('watermark') && Module::isEnabled('watermark')) && isset(Context::getContext()->customer->id)) {
            $type .= '-'.Configuration::get('WATERMARK_HASH');
        }

        // legacy mode or default image
        $theme = ((Shop::isFeatureActive() && file_exists(_PS_PROD_IMG_DIR_.$ids.($type ? '-'.$type : '').'-'.(int) Context::getContext()->shop->id_theme.'.jpg')) ? '-'.Context::getContext()->shop->id_theme : '');
        if ((Configuration::get('PS_LEGACY_IMAGES')
            && (file_exists(_PS_PROD_IMG_DIR_.$ids.($type ? '-'.$type : '').$theme.'.jpg')))
            || ($not_default = strpos($ids, 'default') !== false)) {
            if ($this->allow == 1 && !$not_default) {
                $uri_path = $ids.($type ? '-'.$type : '').$theme.'/'.$name.'.jpg';
            } else {
                $uri_path = _THEME_PROD_DIR_.$ids.($type ? '-'.$type : '').$theme.'.jpg';
            }
        } else {
            // if ids if of the form id_product-id_image, we want to extract the id_image part
            $split_ids = explode('-', $ids);
            $id_image = (isset($split_ids[1]) ? $split_ids[1] : $split_ids[0]);
            $theme = ((Shop::isFeatureActive() && file_exists(_PS_PROD_IMG_DIR_.Image::getImgFolderStatic($id_image).$id_image.($type ? '-'.$type : '').'-'.(int) Context::getContext()->shop->id_theme.'.jpg')) ? '-'.Context::getContext()->shop->id_theme : '');
            if ((int) Configuration::get('PS_REWRITING_SETTINGS') == 1) {
                $uri_path = $id_image.($type ? '-'.$type : '').$theme.'/'.$name.'.jpg';
            } else {
                $uri_path = _THEME_PROD_DIR_.Image::getImgFolderStatic($id_image).$id_image.($type ? '-'.$type : '').$theme.'.jpg';
            }
        }

        return $this->getBaseLink().ltrim($uri_path, '/');
    }

    private function addIndex($xml, $loc, $last_mod)
    {
        $sitemap = $xml->addChild('sitemap');
        $sitemap->addChild('loc', htmlspecialchars($loc));
        $sitemap->addChild('lastmod', $last_mod);

        return $sitemap;
    }

    private function addRow($xml, $loc, $priority, $freq, $last_mod = null)
    {
        if (in_array($loc, $this->removed_links)) {
            return;
        }

        $sitemap = $xml->addChild('url');
        $sitemap->addChild('loc', htmlspecialchars($loc));
        $priority = abs(number_format($priority, 1, '.', ''));

        if ($priority > 1) {
            $priority = '0.'.Tools::substr($priority, -1);
        }

        $sitemap->addChild('priority', $priority);

        if ($last_mod) {
            $sitemap->addChild('lastmod', $last_mod);
        }

        $sitemap->addChild('changefreq', $freq);

        return $sitemap;
    }

    private function addRowImage($xml, $loc, $legend_image)
    {
        $image = $xml->addChild('image', null, 'http://www.google.com/schemas/sitemap-image/1.1');
        $image->addChild('loc', htmlspecialchars($loc), 'http://www.google.com/schemas/sitemap-image/1.1');

        $legend_image = preg_replace('/(&+)/i', '&amp;', $legend_image);

        $image->addChild('caption', $legend_image, 'http://www.google.com/schemas/sitemap-image/1.1');
        $image->addChild('title', $legend_image, 'http://www.google.com/schemas/sitemap-image/1.1');
    }

    private function getAmazzingBlogIndexURI($id_lang, $id_shop, $root_id)
    {
        if (!isset($this->amazzingblog_index_url[$id_lang])) {
            $link_rewrite = $this->db->getValue('SELECT cl.link_rewrite FROM '._DB_PREFIX_.'a_blog_category c
            INNER JOIN '._DB_PREFIX_.'a_blog_category_lang cl
            ON cl.id_category = c.id_category AND cl.id_lang = '.(int) $id_lang.' AND cl.id_shop = '.(int) $id_shop.'
            WHERE c.active = 1 AND c.id_category = '.(int) $root_id);

            if (!empty($link_rewrite)) {
                $this->amazzingblog_index_url[$id_lang] = $this->home_dir.Language::getIsoById($id_lang)."/".$link_rewrite."/";
            } else {
                $this->amazzingblog_index_url[$id_lang] = $this->home_dir;
            }
        }

        return $this->amazzingblog_index_url[$id_lang];
    }
}
