<?php
/**
 *
 *  2018-2018 Kiliba
 *
 *  @author    Kiliba <support@kiliba.com>
 *  @copyright 2018 Kiliba
 *  @license Kiliba
 *
 *  AI in Digital Marketing
 *
 */

require_once(dirname(__FILE__) . '/classes/KilibaLog.php');
require_once(dirname(__FILE__) . '/classes/KilibaApi.php');
require_once(dirname(__FILE__) . '/classes/KilibaAutolog.php');
require_once(dirname(__FILE__) . '/classes/KilibaCache.php');
require_once(dirname(__FILE__) . '/classes/lib/KilibaCaller.php');
require_once(dirname(__FILE__) . '/classes/api/KilibaQueryAbstract.php');
require_once(dirname(__FILE__) . '/classes/api/KilibaQueryCountries.php');
require_once(dirname(__FILE__) . '/classes/api/KilibaQuerySpecificPrices.php');

class Kiliba extends Module
{
    public $tabs = array(
        array(
            'name' => 'Logs Kiliba', // One name for all langs
            'class_name' => 'AdminToLogger',
            'visible' => true,
            'parent_class_name' => 'AdminAdvancedParameters',
        )
    );

    /** @var KilibaCaller */
    public $toCaller;
    protected $schemas;
    public static $moduleName = 'kiliba';

    public function __construct()
    {
        $this->name = 'kiliba';
        $this->tab = 'emailing';
        $this->version = '1.5.2';
        $this->author = 'Kiliba';
        $this->module_key = '8d051f6303e4dfd038895254c8c46888';

        $this->ps_version_compliancy = array('min' => '1.5', 'max' => _PS_VERSION_);
        if (version_compare(_PS_VERSION_, '1.6')  > 0) {
            $this->bootstrap = true;
        }

        $this->toCaller = new KilibaCaller();
        parent::__construct();
        $this->displayName = $this->l('Kiliba');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall the module?');
        $this->description = $this->l('This module is used to retrieve all prestashop data for Kiliba platform');

        /** Load language cache data */
        Language::loadLanguages();

        // LIST OF SCHEMAS USED BY KAFKA
        // Schema of key
        $key_schema=array("type" => "record", "name" => "Key", "fields" => array(array("name" => "id", "type" => "string")));
        // Schema of customers
        $customer_schema=array("type" => "record", "name" => "Customer", "fields" => array(
            array("name" => "email", "type" => "string"),
            array("name" => "id_prestashop", "type" => "string"),
            array("name" => "id_shop_group", "type" => "string"),
            array("name" => "id_shop", "type" => "string"),
            array("name" => "firstname", "type" => "string"),
            array("name" => "lastname", "type" => "string"),
            array("name" => "secure_key", "type" => "string", "default" => ""),
            array("name" => "birthday", "type" => "string"),
            array("name" => "newsletter", "type" => "string"),
            array("name" => "id_gender", "type" => "string"),
            array("name" => "id_lang", "type" => "string"),
            array("name" => "optin", "type" => "string"),
            array("name" => "active", "type" => "string"),
            array("name" => "date_add", "type" => "string"),
            array("name" => "date_update", "type" => "string"),
            array("name" => "id_default_group", "type" => "string", "default" => "0"),
            array("name" => "id_groups", "type" => "string", "default" => ""),
            array("name" => "addresses", "type" =>
                array("type" => "array", "items" => array(
                    "name" => "Address",
                    "type" => "record",
                    "fields" => array(
                        array("name" => "id_address", "type" => "string"),
                        array("name" => "id_country", "type" => "string"),
                        array("name" => "id_state", "type" => "string"),
                        array("name" => "address1", "type" => "string"),
                        array("name" => "address2", "type" => "string"),
                        array("name" => "postcode", "type" => "string"),
                        array("name" => "city", "type" => "string"),
                        array("name" => "phone", "type" => "string"),
                        array("name" => "active", "type" => "string"),
                        array("name" => "deleted", "type" => "string"),
                        array("name" => "date_add", "type" => "string"),
                        array("name" => "date_update", "type" => "string")
                    )
                ))
            )
        ));

        // Schema of visits
        $visit_schema=array("type" => "record", "name" => "Visit", "fields" => array(
            array("name" => "url", "type" => "string"),
            array("name" => "id_customer", "type" => "string"),
            array("name" => "date", "type" => "string"),
            array("name" => "id_product", "type" => "string"),
            array("name" => "id_category", "type" => "string")
        ));

        // Schema of carts
        $cart_schema=array("type" => "record", "name" => "Cart", "fields" => array(
            array("name" => "id", "type" => "string"),
            array("name" => "id_shop_group", "type" => "string"),
            array("name" => "id_shop", "type" => "string"),
            array("name" => "id_customer", "type" => "string"),
            array("name" => "id_currency", "type" => "string"),
            array("name" => "id_address_delivery", "type" => "string"),
            array("name" => "id_address_invoice", "type" => "string"),
            array("name" => "date_add", "type" => "string"),
            array("name" => "date_update", "type" => "string"),
            array("name" => "total_with_tax_with_shipping_with_discount", "type" => "string", "default" => ""),
            array("name" => "total_with_tax_without_shipping_with_discount", "type" => "string", "default" => ""),
            array("name" => "total_discount_with_tax", "type" => "string", "default" => ""),
            array("name" => "total_products_with_tax", "type" => "string", "default" => ""),
            array("name" => "products", "type" =>
                array("type" => "array", "items" => array(
                    "name" => "Product",
                    "type" => "record",
                    "fields" => array(
                        array("name" => "id_product", "type" => "string"),
                        array("name" => "id_product_attribute", "type" => "string"),
                        array("name" => "cart_quantity", "type" => "string"),
                        array("name" => "id_shop", "type" => "string"),
                        array("name" => "reference", "type" => "string"),
                        array("name" => "reduction", "type" => "string"),
                        array("name" => "reduction_type", "type" => "string"),
                        array("name" => "total_wt", "type" => "string"),
                        array("name" => "total_unit_wt", "type" => "string", "default" => ""),
                        array("name" => "id_category_default", "type" => "string")
                    )
                ))
            )
        ));
        // Schema of products
        $product_schema=array("type" => "record", "name" => "Product", "fields" => array(
            array("name" => "id", "type" => "string"),
            array("name" => "id_attribute", "type" => "string"),
            array("name" => "id_shop_default", "type" => "string"),
            array("name" => "on_sale", "type" => "string"),
            array("name" => "id_category_default", "type" => "string"),
            array("name" => "minimal_quantity", "type" => "string"),
            array("name" => "active", "type" => "string"),
            array("name" => "visibility", "type" => "string", "default" => ""),
            array("name" => "wholesale_price", "type" => "string"),
            array("name" => "priority", "type" => "string", "default" => "id_customer;id_shop;id_currency;id_country;id_group"),
            array("name" => "price", "type" => "string"),
            array("name" => "ecotax", "type" => "string", "default" => ""),
            array("name" => "available_for_order", "type" => "string"),
            array("name" => "available_when_out_of_stock", "type" => "string", "default" => ""),
            array("name" => "date_add", "type" => "string"),
            array("name" => "date_update", "type" => "string"),
            array("name" => "quantity", "type" => "string"),
            array("name" => "physical_quantity", "type" => "string"),
            array("name" => "reserved_quantity", "type" => "string"),
            array("name" => "depends_on_stock", "type" => "string"),
            array("name" => "out_of_stock", "type" => "string"),
            array("name" => "image_url", "type" => "string"),
            array("name" => "absolute_url", "type" => "string", "default" => ""),
            array("name" => "relative_url", "type" => "string"),
            array("name" => "default_attribute", "type" => "string"),
            array("name" => "name", "type" => array("type" => "map", "values" => "string")),
            array("name" => "description", "type" => array("type" => "map", "values" => "string")),
            array("name" => "reductions", "type" =>
                array("type" => "array", "items" => array(
                    "name" => "Reduction",
                    "type" => "record",
                    "fields" => array(
                        array("name" => "id_specific_price", "type" => "string"),
                        array("name" => "id_specific_price_rule", "type" => "string"),
                        array("name" => "id_cart", "type" => "string"),
                        array("name" => "id_product", "type" => "string"),
                        array("name" => "id_shop", "type" => "string"),
                        array("name" => "id_shop_group", "type" => "string"),
                        array("name" => "id_currency", "type" => "string"),
                        array("name" => "id_country", "type" => "string"),
                        array("name" => "id_group", "type" => "string"),
                        array("name" => "id_customer", "type" => "string"),
                        array("name" => "id_product_attribute", "type" => "string"),
                        array("name" => "price", "type" => "string"),
                        array("name" => "from_quantity", "type" => "string"),
                        array("name" => "reduction", "type" => "string"),
                        array("name" => "reduction_tax", "type" => "string"),
                        array("name" => "reduction_type", "type" => "string"),
                        array("name" => "from", "type" => "string"),
                        array("name" => "to", "type" => "string")
                    )
                ))
            )

        ));

        $specificprice_schema=array("type" => "record", "name" => "Promotions", "fields" => array(
            array("name" => "id_specific_price", "type" => "string"),
            array("name" => "id_specific_price_rule", "type" => "string"),
            array("name" => "id_cart", "type" => "string"),
            array("name" => "id_product", "type" => "string"),
            array("name" => "id_shop", "type" => "string"),
            array("name" => "id_shop_group", "type" => "string"),
            array("name" => "id_currency", "type" => "string"),
            array("name" => "id_country", "type" => "string"),
            array("name" => "id_group", "type" => "string"),
            array("name" => "id_customer", "type" => "string"),
            array("name" => "id_product_attribute", "type" => "string"),
            array("name" => "price", "type" => "string"),
            array("name" => "from_quantity", "type" => "string"),
            array("name" => "reduction", "type" => "string"),
            array("name" => "reduction_tax", "type" => "string"),
            array("name" => "reduction_type", "type" => "string"),
            array("name" => "from", "type" => "string"),
            array("name" => "to", "type" => "string")
        ));

        $countries_schema=array("type" => "record", "name" => "Pays", "fields" => array(
            array("name" => "id_country", "type" => "string"),
            array("name" => "iso_code", "type" => "string"),
        ));

        // Schema of orders
        $order_schema=array("type" => "record", "name" => "Order", "fields" => array(
            array("name" => "id", "type" => "string"),
            array("name" => "id_shop_group", "type" => "string"),
            array("name" => "id_shop", "type" => "string"),
            array("name" => "id_customer", "type" => "string"),
            array("name" => "id_currency", "type" => "string"),
            array("name" => "id_cart", "type" => "string", "default" => ""),
            array("name" => "valid", "type" => "string", "default" => ""),
            array("name" => "current_state", "type" => "string"),
            array("name" => "reference", "type" => "string"),
            array("name" => "id_address_invoice", "type" => "string"),
            array("name" => "id_address_delivery", "type" => "string"),
            array("name" => "date_add", "type" => "string"),
            array("name" => "date_update", "type" => "string"),
            array("name" => "total_paid", "type" => "string", "default" => ""),
            array("name" => "total_with_tax", "type" => "string", "default" => ""),
            array("name" => "total_without_tax", "type" => "string", "default" => ""),
            array("name" => "total_products_with_tax", "type" => "string", "default" => ""),
            array("name" => "total_products_without_tax", "type" => "string", "default" => ""),
            array("name" => "total_shipping_with_tax", "type" => "string", "default" => ""),
            array("name" => "total_shipping_without_tax", "type" => "string", "default" => ""),
            array("name" => "total_discount_with_tax", "type" => "string", "default" => ""),
            array("name" => "total_discount_without_tax", "type" => "string", "default" => ""),
            array("name" => "products", "type" =>
                array("type" => "array", "items" => array(
                    "name" => "Product",
                    "type" => "record",
                    "fields" => array(
                        array("name" => "id_product", "type" => "string"),
                        array("name" => "id_product_attribute", "type" => "string"),
                        array("name" => "cart_quantity", "type" => "string"),
                        array("name" => "reference", "type" => "string"),
                        array("name" => "reduction_amount", "type" => "string"),
                        array("name" => "reduction_percent", "type" => "string"),
                        array("name" => "group_reduction", "type" => "string"),
                        array("name" => "price", "type" => "string"),
                        array("name" => "total_wt", "type" => "string"),
                        array("name" => "id_category_default", "type" => "string"),
                    )
                ))
            )
        ));

        // array of schemas used by avro
        $this->schemas = array(
            "key" => json_encode($key_schema),
            "customer" => json_encode($customer_schema),
            "visit" => json_encode($visit_schema),
            "cart" => json_encode($cart_schema),
            "product" => json_encode($product_schema),
            "order" => json_encode($order_schema),
            'promotions' => json_encode($specificprice_schema),
            'countries' => json_encode($countries_schema),
        );
    }

    protected function installConfiguration()
    {
        $fluxToken = md5(rand());
        Configuration::updateValue('KILIBA_FLUX_TOKEN', $fluxToken);
        Configuration::updateValue('DONE_SYNCHRO', "0");
        Configuration::updateValue('ID_ACCOUNT_THATSOWL', "");
        Configuration::updateValue('KAFKA_REST_PASSWORD', "");
        Configuration::updateValue('ACTIVE_SYNCHRO', "0");
        Configuration::updateValue('TO_LAST_PRODUCT_ID', 0);
        Configuration::updateValue('TO_LAST_ORDER_ID', 0);
        Configuration::updateValue('TO_LAST_CART_ID', 0);
        Configuration::updateValue('TO_LAST_CUSTOMER_ID', 0);
        Configuration::updateValue('TO_DEBUG_MOD', 0);
        Configuration::updateValue('TO_HOOK_UPDATE_DISABLED', 0);
        Configuration::updateValue('TO_DURATION_SYNCH_INIT', 5);
        Configuration::updateValue('TO_DURATION_SYNCH', 30);
        Configuration::updateValue('TO_BATCH_PRODUCT_NO_ATTRIBUTES', 1);
        Configuration::updateValue('TO_BATCH_SIZE_PRODUCT', 10);
        Configuration::updateValue('TO_BATCH_SIZE_GLOBAL', 50);
        Configuration::updateValue('TO_BATCH_PRODUCT_COUNT', 0);
        Configuration::updateValue('TO_BATCH_ORDER_COUNT', 0);
        Configuration::updateValue('TO_BATCH_CUSTOMER_COUNT', 0);
        Configuration::updateValue('TO_BATCH_CART_COUNT', 0);
        Configuration::updateValue('TO_BATCH_CART_COUNT', 0);
        Configuration::updateValue('TO_CURL_TIMEOUT', 5);
        Configuration::updateValue('TO_CURL_CONNECTTIMEOUT_MS', 5000);
        Configuration::updateValue('TO_CURL_CONNECTTIMEOUT_MS_VISITS', 5000);
        Configuration::updateValue('KILIBA_TO_GENERATE_MODE', 'hook');
        Configuration::updateValue('KILIBA_XML_MINIMAL_INTERVAL', 300);
        Configuration::updateValue('KILIBA_XML_LAST_CALL', 0);
        Configuration::updateValue('KILIBA_XML_NB_PRODUCT_ITERATOR', 150);

        foreach (Shop::getShops(true, null, true) as $id_shop) {
            Configuration::updateValue('KILIBA_FLUX_TOKEN', $fluxToken, false, null, $id_shop);
            Configuration::updateValue('DONE_SYNCHRO', "0", false, null, $id_shop);
            Configuration::updateValue('ID_ACCOUNT_THATSOWL', "", false, null, $id_shop);
            Configuration::updateValue('KAFKA_REST_PASSWORD', "", false, null, $id_shop);
            Configuration::updateValue('ACTIVE_SYNCHRO', "0", false, null, $id_shop);
            Configuration::updateValue('TO_LAST_PRODUCT_ID', 0, false, null, $id_shop);
            Configuration::updateValue('TO_LAST_ORDER_ID', 0, false, null, $id_shop);
            Configuration::updateValue('TO_LAST_CART_ID', 0, false, null, $id_shop);
            Configuration::updateValue('TO_LAST_CUSTOMER_ID', 0, false, null, $id_shop);
            Configuration::updateValue('TO_DEBUG_MOD', 0, false, null, $id_shop);
            Configuration::updateValue('TO_HOOK_UPDATE_DISABLED', 0, false, null, $id_shop);
            Configuration::updateValue('TO_DURATION_SYNCH_INIT', 5, false, null, $id_shop);
            Configuration::updateValue('TO_DURATION_SYNCH', 30, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_PRODUCT_NO_ATTRIBUTES', 1, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_SIZE_PRODUCT', 10, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_SIZE_GLOBAL', 50, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_PRODUCT_COUNT', 0, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_ORDER_COUNT', 0, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_CUSTOMER_COUNT', 0, false, null, $id_shop);
            Configuration::updateValue('TO_BATCH_CART_COUNT', 0, false, null, $id_shop);
            Configuration::updateValue('TO_CURL_TIMEOUT', 5, false, null, $id_shop);
            Configuration::updateValue('TO_CURL_CONNECTTIMEOUT_MS', 5000, false, null, $id_shop);
            Configuration::updateValue('TO_CURL_CONNECTTIMEOUT_MS_VISITS', 5000, false, null, $id_shop);
            Configuration::updateValue('KILIBA_TO_GENERATE_MODE', 'hook', false, null, $id_shop);
            Configuration::updateValue('KILIBA_XML_MINIMAL_INTERVAL', 300, false, null, $id_shop);
            Configuration::updateValue('KILIBA_XML_LAST_CALL', 0, false, null, $id_shop);
            Configuration::updateValue('KILIBA_XML_NB_PRODUCT_ITERATOR', 150, false, null, $id_shop);
            $id_customer_group = Configuration::get('PS_CUSTOMER_GROUP', null, null, $id_shop);
            if (empty($id_customer_group)) {
                continue;
            }

            $group = new Group($id_customer_group);
            Configuration::updateValue('TO_PRICE_WITHOUT_TAX', $group->price_display_method, null, null, $id_shop);
        }

        return true;
    }

    protected function installHook()
    {
        return $this->registerHook('displayHeader') &&
            $this->registerHook('displayFooter') &&
            $this->registerHook('actionValidateCustomerAddressForm') &&
            $this->registerHook('displayOrderConfirmation') &&
            $this->registerHook('actionCustomerAccountUpdate') &&
            $this->registerHook('actionCartSave') &&
            $this->registerHook('updateProduct') &&
            $this->registerHook('actionAttributePostProcess') &&
            $this->registerHook('displayFeatureValuePostProcess') &&
            $this->registerHook('actionOrderStatusUpdate') &&
            $this->registerHook('adminCustomers') &&
            $this->registerHook('actionObjectUpdateAfter') &&
            $this->registerHook('actionAdminOrdersTrackingNumberUpdate') &&
            $this->registerHook('actionObjectDeleteAfter') &&
            $this->registerHook('actionObjectAddAfter');
    }

    protected function installSql()
    {
        $sql = array();
        $return = true;
        include(dirname(__FILE__) . '/sql/sql-install.php');
        foreach ($sql as $s) {
            if (!Db::getInstance()->execute($s)) {
                $return = false;
            }
        }

        return $return;
    }

    protected function createTab($id_parent, $module, $name, $class_name, $status = true)
    {
        $tab = new Tab();
        $tab->module = $module;

        foreach (Language::getLanguages(true) as $languages) {
            $tab->name[$languages['id_lang']] = $name;
        }

        $tab->id_parent = $id_parent;
        $tab->class_name = $class_name;
        $tab->active = $status;
        $r = $tab->save();

        if ($r === false) {
            return false;
        }

        return $tab->id;
    }

    protected function createAdminTabs()
    {
        $className = 'AdminInformation';
        if (version_compare(_PS_VERSION_, '1.7.0.0') > 0) {
            $className = 'AdminAdvancedParameters';
        } elseif (version_compare(_PS_VERSION_, '1.6') < 0) {
            $className = 'AdminTools';
        }

        $parent_tab = Db::getInstance()->getValue('SELECT id_tab FROM '._DB_PREFIX_.'tab WHERE id_parent = 0 AND class_name = "'.$className.'"');
        return (
        $this->createTab($parent_tab, $this->name, $this->l('Logs Kiliba'), 'AdminToLogger')
        );
    }

    public function install()
    {
        if (!parent::install()
            || !$this->installConfiguration()
            || !$this->createAdminTabs()
            || !$this->installHook()
            || !$this->installSql()) {
            $this->uninstall();
            return false;
        }

        return true;
    }

    protected function uninstallConfiguration()
    {
        Configuration::deleteByName('KILIBA_FLUX_TOKEN');
        Configuration::deleteByName('ID_ACCOUNT_THATSOWL');
        Configuration::deleteByName('ACTIVE_SYNCHRO');
        Configuration::deleteByName('DONE_SYNCHRO');
        Configuration::deleteByName('KAFKA_REST_PASSWORD');
        Configuration::deleteByName('PAGES_PRODUCT');
        Configuration::deleteByName('PAGES_ORDER');
        Configuration::deleteByName('PAGES_CUSTOMER');
        Configuration::deleteByName('PAGES_CART');
        Configuration::deleteByName('TO_HOOK_UPDATE_DISABLED');
        Configuration::deleteByName('TO_DURATION_SYNCH_INIT');
        Configuration::deleteByName('TO_DURATION_SYNCH');
        Configuration::deleteByName('TO_BATCH_PRODUCT_NO_ATTRIBUTES');
        Configuration::deleteByName('TO_BATCH_SIZE_PRODUCT');
        Configuration::deleteByName('TO_BATCH_SIZE_GLOBAL');
        Configuration::deleteByName('TO_BATCH_COUNTRIES_DO');
        Configuration::deleteByName('TO_BATCH_SPECIFICPRICE_DO');
        Configuration::deleteByName('TO_BATCH_CART_DO');
        Configuration::deleteByName('TO_BATCH_ORDER_DO');
        Configuration::deleteByName('TO_BATCH_CUSTOMER_DO');
        Configuration::deleteByName('TO_BATCH_PRODUCT_DO');
        Configuration::deleteByName('TO_BATCH_COUNTRIES_COUNT');
        Configuration::deleteByName('TO_BATCH_SPECIFICPRICE_COUNT');
        Configuration::deleteByName('TO_LAST_COUNTRIES_ID');
        Configuration::deleteByName('TO_LAST_SPECIFICPRICE_ID');
        Configuration::deleteByName('TO_CACHE_KEY_TOTAL_ORDERS_DATE');
        Configuration::deleteByName('TO_CACHE_KEY_TOTAL_ORDERS');
        Configuration::deleteByName('TO_TOKEN_SYNC');
        Configuration::deleteByName('TO_PRICE_WITHOUT_TAX');
        Configuration::deleteByName('TO_CURL_CONNECTTIMEOUT_MS_VISITS');
        Configuration::deleteByName('TO_CURL_CONNECTTIMEOUT_MS');
        Configuration::deleteByName('TO_CURL_TIMEOUT');
        Configuration::deleteByName('TO_BATCH_CART_COUNT');
        Configuration::deleteByName('TO_BATCH_ORDER_COUNT');
        Configuration::deleteByName('TO_BATCH_CUSTOMER_COUNT');
        Configuration::deleteByName('TO_BATCH_PRODUCT_COUNT');
        Configuration::deleteByName('TO_LAST_CUSTOMER_ID');
        Configuration::deleteByName('TO_LAST_CART_ID');
        Configuration::deleteByName('TO_LAST_ORDER_ID');
        Configuration::deleteByName('TO_LAST_PRODUCT_ID');
        Configuration::deleteByName('KILIBA_TO_GENERATE_MODE');
        Configuration::deleteByName('KILIBA_XML_MINIMAL_INTERVAL');
        Configuration::deleteByName('KILIBA_XML_LAST_CALL');
        Configuration::deleteByName('KILIBA_XML_NB_PRODUCT_ITERATOR');
    }

    protected function uninstallTab($tabClass)
    {
        if (version_compare(_PS_VERSION_, '1.7.1.0') > 0) {
            return true;
        }

        $idTab = Tab::getIdFromClassName($tabClass);
        if ($idTab != 0) {
            $tab = new Tab($idTab);
            $tab->delete();
            return true;
        }

        return true;
    }

    protected function uninstallSql()
    {
        $sqlu = array();
        $return = true;
        include(dirname(__FILE__) . '/sql/sql-uninstall.php');
        foreach ($sqlu as $s) {
            if (!Db::getInstance()->execute($s)) {
                $return = false;
            }
        }

        return $return;
    }

    public function uninstall()
    {
        $this->uninstallConfiguration();
        if (!parent::uninstall()
            || !$this->uninstallSql()
            || !$this->uninstallTab('AdminToLogger')) {
            return false;
        }

        return true;
    }

    protected static function getBaseLink($shop, $relative_protocol = false)
    {
        $ssl = (ConfigurationCore::get('PS_SSL_ENABLED', null, null, Context::getContext()->shop->id)
            && Configuration::get('PS_SSL_ENABLED_EVERYWHERE', null, null, Context::getContext()->shop->id));

        if ($relative_protocol) {
            $base = '//'.($ssl ? $shop->domain_ssl : $shop->domain);
        } else {
            $base = ($ssl ? 'https://'.$shop->domain_ssl : 'http://'.$shop->domain);
        }

        return $base.$shop->getBaseURI();
    }

    /**
     * GET URL TO USE FOR SYNCHRONISATION
     */
    public static function getCronUrl($tokenSync, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $token = Configuration::get('KILIBA_FLUX_TOKEN');

        $products = (string)Configuration::get('TO_LAST_PRODUCT_ID', false, null, $shop->id);
        $orders = (string)Configuration::get('TO_LAST_ORDER_ID', false, null, $shop->id);
        $carts = (string)Configuration::get('TO_LAST_CART_ID', false, null, $shop->id);
        $customers = (string)Configuration::get('TO_LAST_CUSTOMER_ID', false, null, $shop->id);
        $specificprices = (string)Configuration::get('TO_LAST_SPECIFICPRICE_ID', false, null, $shop->id);
        $countries = (string)Configuration::get('TO_LAST_COUNTRIES_ID', false, null, $shop->id);
        if (!$products) {
            $products = 0;
        }
        if (!$orders) {
            $orders = 0;
        }
        if (!$carts) {
            $carts = 0;
        }
        if (!$customers) {
            $customers = 0;
        }
        if (!$specificprices) {
            $specificprices = 0;
        }
        if (!$countries) {
            $countries = 0;
        }

        $params = array(
            'token' => $token,
            'products' => $products,
            'carts' => $carts,
            'orders' => $orders,
            'customers' => $customers,
            'specificprices' => $specificprices,
            'countries' => $countries,
            'tokenSync' => $tokenSync
        );


        $uri = Context::getContext()->link->getModuleLink(self::$moduleName, 'cron', $params, null, null, $shop->id);

        return array("url" => $uri) ;
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        ini_set('max_execution_time', 0);

        $output = '';
        if (Module::getModuleIdByName('thatsowl') || Module::isInstalled('thatsowl')) {
            $output .= $this->displayError($this->l('Thatsowl module exists in your shop, please delete him before use new Kiliba module'));
            return $output;
        }

        if (Shop::getContext() != Shop::CONTEXT_SHOP) {
            $output .= $this->displayError($this->l('You must select only one shop in shop dropdown'));
            return $output;
        }

        if (((bool) Tools::isSubmit('submitConfig')) == true) { // If values have been submitted in the form, process.
            $result = $this->postProcess();
            if ("$result" == "1") {
                $output .= $this->displayConfirmation($this->l('Successful update'));
            } else {
                $this->toCaller->sendRequest("$result");
                $errorMessage = "";
                switch ("$result") {
                    case "DONE_SYNCHRO":
                        $errorMessage = $this->l('SYNCHRONIZATION HAS ALREADY BEEN DONE');//LA SYNCHRONISATION A DEJA ETE REALISEE
                        $output .= $this->displayConfirmation($this->l('SYNCHRONIZATION HAS ALREADY BEEN DONE'));
                        break;
                    case "NO_ACCOUNT":
                        $errorMessage = $this->l('INCORRECT ID');//ID INCORRECT
                        $output .= $this->displayError($errorMessage);
                        break;
                    case "INACTIVE_ACCOUNT":
                        $errorMessage = $this->l('PLEASE ACTIVATE YOUR ACCOUNT');//VEUILLEZ ACTIVER VOTRE COMPTE
                        $output .= $this->displayError($errorMessage);
                        break;
                    case "ACTIVE_SYNCHRO":
                        $errorMessage = $this->l('SYNCHRONIZATION IS ALREADY IN PROGRESS');//LA SYNCHRONISATION EST DEJA EN COURS D'EXECUTION
                        $output .= $this->displayConfirmation($this->l('SYNCHRONIZATION IS ALREADY IN PROGRESS'));
                        break;
                    default:
                        $errorMessage = $this->l('AN ERROR OCCURRED');//UNE ERREUR EST SURVENUE
                        $output .= $this->displayError($errorMessage);
                        break;
                }
            }
        }
        return $output.$this->displayForm();
    }


    /**
     * Create the form that will be displayed in the configuration of module.
     */
    protected function displayForm()
    {
        $defaultLang = (int)Configuration::get('PS_LANG_DEFAULT');
        // Init Fields form array
        $fields_form = array();
        $fields_form[0]['form'] = array(
            'legend' => array(
                'title' => $this->l('Settings'),
                'icon'  => 'icon-cogs',
            ),
            'input'  => array(
                array(
                    'type'  => 'free',
                    'name'  => 'THATSOWL_TOOLTIP'
                ),
                array(
                    'type'  => 'text',
                    'name'  => 'ID_ACCOUNT_THATSOWL',
                    'label' => $this->l('Add your Kiliba AccountId') . ' : ',
                    'col'   => 1,
                    'size' => 20,
                    'required' => true
                ),
                array(
                    'type'  => 'free',
                    'name'  => 'THATSOWL_URL_LOGS',
                    'label' => $this->l('Logs') . ' : '
                ),
                array(
                    'type'  => 'free',
                    'name'  => 'THATSOWL_TOKEN',
                    'label' => $this->l('Token prestashop') . ' : '),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
                'class' => 'btn btn-default pull-right button',
                'name' => 'submitConfig',
            ),
        );

        $helper = new HelperForm();

        // Module, token and currentIndex
        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        // Language
        $helper->default_form_language = $defaultLang;
        $helper->allow_employee_form_lang = $defaultLang;

        // Title and toolbar
        $helper->title = $this->displayName;
        $helper->show_toolbar = true;        // false -> remove toolbar
        $helper->toolbar_scroll = true;      // yes - > Toolbar is always visible on the top of the screen.
        $helper->submit_action = 'submit' . $this->name;

        $helper->table                    = $this->table;
        $helper->module                   = $this;

        $fields_values = $this->getConfigFieldsValues();
        $fields_values['THATSOWL_URL_LOGS'] = '<div style="font-size: 13px; width: fit-content;" ><a href="'.Context::getContext()->link->getAdminLink('AdminToLogger', true).'">'.Context::getContext()->link->getAdminLink('AdminToLogger', true).'</a></div>';
        $fields_values['THATSOWL_TOKEN'] = '<div style="font-size: 13px; width: fit-content;">'.Configuration::get('KILIBA_FLUX_TOKEN', null, null, Context::getContext()->shop->id).'</div>';
        $fields_values['THATSOWL_TOOLTIP'] = '<div style="font-size: 13px; width: fit-content;">Votre AccountId vous est fourni à la création de votre compte sur <a href="https:\/\/app.thatsowl.com">https://app.thatsowl.com</a></div>';
        $helper->tpl_vars = array(
            'fields_value' => $fields_values, /* Add values for inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language'  => $this->context->language->id,
        );
        $helper->fields_value['kiliba'] = Configuration::get('kiliba');

        return $helper->generateForm($fields_form);
    }


    /**
     * Set values for the inputs.
     */
    protected function getConfigFieldsValues()
    {
        return array(
            'ID_ACCOUNT_THATSOWL' => Tools::getValue('ID_ACCOUNT_THATSOWL', ConfigurationCore::get('ID_ACCOUNT_THATSOWL', null, null, Context::getContext()->shop->id)),
        );
    }

    /**
     * PROCESS FORM DATA
     */
    protected function postProcess()
    {
        $shop = Context::getContext()->shop;

        ini_set('max_execution_time', 0);
        ini_set("default_socket_timeout", 7200);
        $form_values = $this->getConfigFieldsValues();
        $id_to_check = "";
        foreach (array_keys($form_values) as $key) {
            if ($key != "ID_ACCOUNT_THATSOWL") {
                Configuration::updateValue($key, Tools::getValue($key), false, null, $shop->id);
            } else {
                $id_to_check = Tools::getValue($key);
            }
        }


        $id_Thatsowl = $id_to_check;
        // CHECK IF ACCOUNT CAN BE SYNCED
        $result = $this->toCaller->checkForm($id_to_check);
        // if ($result['success'] === false) {
        //     return "Une erreur est survenue";
        // }

        $response = $result['response'];
        if ($response['result'] != "1" && $response['message'] != "DONE_SYNCHRO" && $response['message'] != "ACTIVE_SYNCHRO") {
            return $response['message'];
        }
        Configuration::updateValue('ID_ACCOUNT_THATSOWL', $id_Thatsowl, false, null, $shop->id);

        $password = $response['password'];
        // PASSWORD USED FOR BASIC AUTH IN KAFKA
        Configuration::updateValue('KAFKA_REST_PASSWORD', $password, false, null, $shop->id);

        if ($response['result'] != "1" && $response['message'] == "DONE_SYNCHRO") {
            Configuration::updateValue('DONE_SYNCHRO', "1", false, null, $shop->id);
            return $response['message'];
        }

        // SEND LAST PRODUCTS
        $last_products = $this->getLastProducts($shop);
        $records = array();
        foreach ($last_products as $product) {
            $records_to_add = $this->prepareProduct($product["id_product"], $shop);
            $records = array_merge($records, $records_to_add);
        }
        // SEND 5 LAST PRODUCTS
        $this->toCaller->postKafka("products", $this->schemas['product'], $this->schemas['key'], $records, true, true, $shop);

        // INIT SYNC
        $this->toCaller->callSync(null, $shop);

        return true;
    }

    /**
     * Retrieve data on installation Sync and send it.
     */
    public function installationSync($params)
    {
        // ini_set('max_execution_time', 0);
        // $max_time = ini_get('max_execution_time');

        $shop = Context::getContext()->shop;

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);//Retrieve image url (not full link but restrein link)
        if ($id_account == "") {
            return array(
                'error' => 'Please enter a valid code in order to start the synchronisation.',
                'errorCode' => 'error_account_thatsowl',
                'success' => false,
                'payload' => array()
            );
        }

        $duration_sync = (int)Configuration::get('TO_DURATION_SYNCH', null, null, $shop->id) ;
        $max_execution_time = ini_get("max_execution_time");

        $duration_limit = $max_execution_time - 10;
        if (!$duration_sync) {
            $duration_sync = 30;
        }
        if ($duration_sync > $duration_limit && $duration_limit > 0) {
            $duration_sync = $duration_limit;
        }

        // $this->toCaller->sendRequest("DURATION SYNC : ".(string)$duration_sync);

        $idsToTreat = $params["idsToTreat"];
        $last_id_customer = (int)$idsToTreat["customers"];
        $last_id_order = (int)$idsToTreat["orders"];
        $last_id_product = (int)$idsToTreat["products"];
        $last_id_cart = (int)$idsToTreat["carts"];
        $last_id_specificprice = (int)$idsToTreat["specificprices"];
        $last_id_country = (int)$idsToTreat["countries"];

        $idsTreated = $idsToTreat;
        $logs = array();

        $begin = time();
        $end = time();
        $errorsInRow = 0;
        $maxErrorsInRow = 10;

        $batchSize = 50;
        if (!!Configuration::get('TO_BATCH_SIZE_GLOBAL', null, null, $shop->id)) {
            $batchSize = Configuration::get('TO_BATCH_SIZE_GLOBAL', null, null, $shop->id);
        }

        // CHECK QUANTITY OF DATA TO PROCESS
        $querySpecificPrices = new KilibaQuerySpecificPrices($shop);
        $queryCountries = new KilibaQueryCountries($shop);

        $total_orders = KilibaCache::getCached('TOTAL_ORDERS', function () {
            return (int)Kiliba::getTotalNoOfTable('orders');
        });
        $total_products = KilibaCache::getCached('TOTAL_PRODUCTS', function () {
            return (int)Kiliba::getTotalNoOfTable('product');
        });
        $total_customers = KilibaCache::getCached('TOTAL_CUSTOMERS', function () {
            return (int)Kiliba::getTotalNoOfTable('customer');
        });
        $total_carts = KilibaCache::getCached('TOTAL_CARTS', function () {
            return (int)Kiliba::getTotalNoOfTable('cart');
        });
        $total_specificprices = $querySpecificPrices->count();
        $total_countries = $queryCountries->count();

        if ($params["begin"] === true) {
            // IF FULL SYNC (every tables) => checkBeforeStartSync
            if ($last_id_product > -1 && $last_id_customer > -1 && $last_id_order > -1 && $last_id_cart > -1 && $last_id_specificprice > -1 && $last_id_country > -1) {
                $response = $this->toCaller->checkBeforeStartSync($shop);//Check if sync can be started
                if (!$response['success']) {
                    return array(
                      'error' => 'Serveur indisponible lors du startsync : '.$response['httpCode'].' '.$response['error'],
                      'errorCode' => 'error_startsync',
                      'success' => false,
                      'payload' => array()
                        );
                } else {
                    // FORCEMENT customer=0&order=0&product=0&cart=0 CAR BEGIN=true
                    Configuration::updateValue('TO_LAST_CART_ID', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_LAST_PRODUCT_ID', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_LAST_CUSTOMER_ID', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_LAST_ORDER_ID', 0, null, null, $shop->id);
                    $querySpecificPrices->setLastId(0);
                    $queryCountries->setLastId(0);

                    Configuration::updateValue('TO_BATCH_PRODUCT_COUNT', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_CUSTOMER_COUNT', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_ORDER_COUNT', 0, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_CART_COUNT', 0, null, null, $shop->id);
                    $querySpecificPrices->setCount(0);
                    $queryCountries->setCount(0);
                }
            } else {
                // FORCEMENT customer=-1 || order=-1  || product=-1  || cart=1
                Configuration::updateValue('TO_LAST_CART_ID', ($last_id_cart > -1 ? $last_id_cart : $total_carts), null, null, $shop->id);
                Configuration::updateValue('TO_LAST_PRODUCT_ID', ($last_id_product > -1 ? $last_id_product : $total_products), null, null, $shop->id);
                Configuration::updateValue('TO_LAST_CUSTOMER_ID', ($last_id_customer > -1 ? $last_id_customer : $total_customers), null, null, $shop->id);
                Configuration::updateValue('TO_LAST_ORDER_ID', ($last_id_order > -1 ? $last_id_order : $total_orders), null, null, $shop->id);
                $querySpecificPrices->setLastId(($last_id_specificprice > -1 ? $last_id_specificprice : $total_specificprices));
                $queryCountries->setLastId(($last_id_country > -1 ? $last_id_country : $total_countries));

                Configuration::updateValue('TO_BATCH_PRODUCT_COUNT', ($last_id_product !== 0 ? $this->getCountOfTable('product', $last_id_product) : 0), null, null, $shop->id);
                Configuration::updateValue('TO_BATCH_CUSTOMER_COUNT', ($last_id_customer !== 0 ? $this->getCountOfTable('customer', $last_id_customer) : 0), null, null, $shop->id);
                Configuration::updateValue('TO_BATCH_ORDER_COUNT', ($last_id_order !== 0 ? $this->getCountOfTable('orders', $last_id_order) : 0), null, null, $shop->id);
                Configuration::updateValue('TO_BATCH_CART_COUNT', ($last_id_cart !== 0 ? $this->getCountOfTable('cart', $last_id_cart) : 0), null, null, $shop->id);
                $querySpecificPrices->setCount(($last_id_specificprice !== 0 ? $querySpecificPrices->count($last_id_specificprice) : 0));
                $queryCountries->setCount(($last_id_country !== 0 ? $queryCountries->count($last_id_country) : 0));
            }
            Configuration::updateValue('TO_BATCH_PRODUCT_DO', $last_id_product > -1, null, null, $shop->id);
            Configuration::updateValue('TO_BATCH_CUSTOMER_DO', $last_id_customer > -1, null, null, $shop->id);
            Configuration::updateValue('TO_BATCH_ORDER_DO', $last_id_order > -1, null, null, $shop->id);
            Configuration::updateValue('TO_BATCH_CART_DO', $last_id_cart > -1, null, null, $shop->id);
            $querySpecificPrices->setFinished($last_id_specificprice === -1);
            $queryCountries->setFinished($last_id_country === -1);
            $duration_sync = 5;
        }

        $totalsToTreat = array(
            "products" => $total_products,
            "orders" => $total_orders,
            "carts" => $total_carts,
            "customers" => $total_customers,
            "specificprices" => $total_specificprices,
            "countries" => $total_countries
        );

        if ($params["begin"] === true) {
            $this->toCaller->sendRequest("Synchro Starting");
            $logs[] = 'Synchro Starting';
        } else {
            $this->toCaller->sendRequest("Batch Starting");
            $logs[] = 'Batch Starting';
        }

        $count_customer = (int)Configuration::get('TO_BATCH_CUSTOMER_COUNT', null, null, $shop->id);
        $count_cart = (int)Configuration::get('TO_BATCH_CART_COUNT', null, null, $shop->id);
        $count_order = (int)Configuration::get('TO_BATCH_ORDER_COUNT', null, null, $shop->id);
        $count_product = (int)Configuration::get('TO_BATCH_PRODUCT_COUNT', null, null, $shop->id);
        $count_specificprice = $querySpecificPrices->getCount();
        $count_country = $queryCountries->getCount();

        $string = 'TOTAL ALREADY TREATED customers: '.$count_customer.'/'.$total_customers.', orders: '.$count_order.'/'.$total_orders.', products: '.$count_product.'/'.$total_products.', carts: '.$count_cart.'/'.$total_carts.', specificprices: '.$count_specificprice.'/'.$total_specificprices.', countries: '.$count_country.'/'.$total_countries.'';
        $this->toCaller->sendRequest($string);
        $logs[] = $string;

        $alreadyTreated = array(
            "products" => $count_product,
            "orders" => $count_order,
            "carts" => $count_cart,
            "customers" => $count_customer,
            'specificprices' => $count_specificprice,
            'countries' => $count_country,
        );
        // WHILE DURATION IS RESPECTED DO SYNC

        $do_customer = (bool)Configuration::get('TO_BATCH_CUSTOMER_DO', null, null, $shop->id);
        $do_cart = (bool)Configuration::get('TO_BATCH_CART_DO', null, null, $shop->id);
        $do_order = (bool)Configuration::get('TO_BATCH_ORDER_DO', null, null, $shop->id);
        $do_product = (bool)Configuration::get('TO_BATCH_PRODUCT_DO', null, null, $shop->id);
        $do_specificprice = (bool)!$querySpecificPrices->isFinished();
        $do_country = (bool)!$queryCountries->isFinished();

        $memory_limit = ini_get('memory_limit');
        if (empty($memory_limit)) {
            $memory_limit = '126M';
        }
        if ($memory_limit == '-1') {
            $memory_limit = '126M';
        }
        if (preg_match('/^(\d+)(.)$/', $memory_limit, $matches)) {
            if ($matches[2] == 'G') {
                $memory_limit = $matches[1] * 1024 * 1024 * 1024; // nnnM -> nnn MB
            } elseif ($matches[2] == 'M') {
                $memory_limit = $matches[1] * 1024 * 1024; // nnnM -> nnn MB
            } elseif ($matches[2] == 'K') {
                $memory_limit = $matches[1] * 1024; // nnnK -> nnn KB
            }
        }
        $memory_limit = $memory_limit-1000000;

        while (($end - $begin) < $duration_sync && memory_get_peak_usage() <= $memory_limit) {
            if (!$queryCountries->isFinished()) {
                $queryCountries->execute();
                $errorsInRow = (int)$queryCountries->hasError();
                if ($errorsInRow) {
                    break;
                }
                if (!$queryCountries->hasNextResults()) {
                    $queryCountries->setFinished();
                }
                $count_country = $queryCountries->getCount();
                $end = time();
            }

            if (!$querySpecificPrices->isFinished()) {
                $querySpecificPrices->execute();
                $errorsInRow = (int)$querySpecificPrices->hasError();
                if ($errorsInRow) {
                    break;
                }
                if (!$querySpecificPrices->hasNextResults()) {
                    $querySpecificPrices->setFinished();
                }
                $count_specificprice = $querySpecificPrices->getCount();
                $end = time();
            }

            if ($do_customer) {
                $customers = $this->getDBById('customer', $last_id_customer, $batchSize);
                if (!empty($customers)) {
                    $success = false;

                    $topic = "customers";
                    $records = array();

                    $count_customer_before = $count_customer;
                    $last_id_customer_before = $last_id_customer;
                    foreach ($customers as $customer) {
                        $record = $this->prepareCustomer($customer['id_customer']);
                        $count_customer++;
                        if (false === $record) {
                            continue;
                        }
                        array_push($records, $record);
                    }
                    $last_id_customer = (int)$customer['id_customer'];

                    while ($errorsInRow < $maxErrorsInRow && $success == false) {
                        $result = $this->toCaller->postKafka($topic, $this->schemas['customer'], $this->schemas['key'], $records, true);
                        if ($result == false) {
                            $errorsInRow = $errorsInRow + 1;
                            sleep(1);
                            continue;
                        }

                        $success = true;
                        $errorsInRow = 0;
                    }

                    if (!$success) {
                        $count_customer = $count_customer_before;
                        $last_id_customer = $last_id_customer_before;
                        break;
                    }

                    // SEND SYNCHRO PROGRESS
                    sleep(1);
                    $end = time();

                    Configuration::updateValue('TO_LAST_CUSTOMER_ID', $last_id_customer, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_CUSTOMER_COUNT', $count_customer, null, null, $shop->id);
                    continue;
                }

                $do_customer = false;
                Configuration::updateValue('TO_BATCH_CUSTOMER_DO', $do_customer, null, null, $shop->id);
            }

            if ($do_product) {
                $products = $this->getDBById('product', $last_id_product, $batchSize);
                if (!empty($products)) {
                    $success = false;
                    $records = array();
                    $topic = "products";
                    $count_product_before = $count_product;
                    $last_id_product_before = $last_id_product;
                    foreach ($products as $product) {
                        $records_to_add = $this->prepareProduct($product["id_product"]);
                        $records = array_merge($records, $records_to_add);
                        $count_product++;
                    }
                    $last_id_product = (int)$product['id_product'];
                    while ($errorsInRow < $maxErrorsInRow && $success == false) {
                        $result = $this->toCaller->postKafka($topic, $this->schemas['product'], $this->schemas['key'], $records, true);
                        if ($result == false) {
                            $errorsInRow = $errorsInRow + 1;
                            sleep(1);
                            continue;
                        }
                        $success = true;
                        $errorsInRow = 0;
                    }
                    if (!$success) {
                        $count_product = $count_product_before;
                        $last_id_product = $last_id_product_before;
                        break;
                    }
                    // SEND SYNCHRO PROGRESS
                    sleep(1);
                    $end = time();
                    Configuration::updateValue('TO_LAST_PRODUCT_ID', $last_id_product, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_PRODUCT_COUNT', $count_product, null, null, $shop->id);
                    continue;
                }
                $do_product = false;
                Configuration::updateValue('TO_BATCH_PRODUCT_DO', $do_product, null, null, $shop->id);
            }

            if ($do_cart) {
                $carts = $this->getDBById('cart', $last_id_cart, $batchSize);
                if (!empty($carts)) {
                    $success = false;

                    $topic = "carts";
                    $records = array();

                    $count_cart_before = $count_cart;
                    $last_id_cart_before = $last_id_cart;
                    foreach ($carts as $cart) {
                        $cart = new Cart($cart['id_cart']);

                        Context::getContext()->cart = $cart;
                        $record = $this->prepareCart($cart);

                        $count_cart++;
                        if ($record === false) {
                            continue;
                        }
                        array_push($records, $record);
                    }
                    $last_id_cart = (int)$cart->id;

                    while ($errorsInRow < $maxErrorsInRow && $success == false) {
                        $result = $this->toCaller->postKafka($topic, $this->schemas['cart'], $this->schemas['key'], $records, true);
                        if ($result == false) {
                            $errorsInRow = $errorsInRow + 1;
                            sleep(1);
                            continue;
                        }

                        $success = true;
                        $errorsInRow = 0;
                    }

                    if (!$success) {
                        $count_cart = $count_cart_before;
                        $last_id_cart = $last_id_cart_before;
                        break;
                    }

                    // SEND SYNCHRO PROGRESS
                    sleep(1);
                    $end = time();

                    Configuration::updateValue('TO_LAST_CART_ID', $last_id_cart, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_CART_COUNT', $count_cart, null, null, $shop->id);
                    continue;
                }

                $do_cart = false;
                Configuration::updateValue('TO_BATCH_CART_DO', $do_cart, null, null, $shop->id);
            }

            // ORDERS ARE DONE AT LAST BECAUSE THEY TAKE MORE TIME TO BE TREATED.
            if ($do_order) {
                $orders = $this->getDBById('orders', $last_id_order, $batchSize);
                if (!empty($orders)) {
                    $success = false;
                    $topic = "orders";

                    $records = array();
                    $count_order_before = $count_order;
                    $last_id_order_before = $last_id_order;
                    foreach ($orders as $order) {
                        $objOrder = new Order((int)$order['id_order']);
                        $record = $this->prepareOrder($objOrder);
                        $count_order++;
                        if (false === $record) {
                            continue;
                        }
                        array_push($records, $record);
                    }
                    $last_id_order = (int)$order['id_order'];

                    while ($errorsInRow < $maxErrorsInRow && $success == false) {
                        $result = $this->toCaller->postKafka($topic, $this->schemas['order'], $this->schemas['key'], $records, true);
                        if ($result == false) {
                            $errorsInRow = $errorsInRow + 1;
                            sleep(1);
                            continue;
                        }

                        $success = true;
                        $errorsInRow = 0;
                    }

                    if (!$success) {
                        $last_id_order = $last_id_order_before;
                        $count_order = $count_order_before;
                        break;
                    }

                    // SEND SYNCHRO PROGRESS
                    sleep(1);
                    $end = time();

                    Configuration::updateValue('TO_LAST_ORDER_ID', $last_id_order, null, null, $shop->id);
                    Configuration::updateValue('TO_BATCH_ORDER_COUNT', $count_order, null, null, $shop->id);
                    continue;
                }

                $do_order = false;
                Configuration::updateValue('TO_BATCH_ORDER_DO', $do_order, null, null, $shop->id);
            }

            break;
        }
        if (($end - $begin) >= $duration_sync) {
            $logs[] = 'Limit of time reached : '.($end - $begin).'/'.$duration_sync;
        }
        if (memory_get_peak_usage() > $memory_limit) {
            $logs[] = 'Memory limit reached : '.memory_get_peak_usage().'/'.$memory_limit;
        }

        $treated = array(
            "products" => $count_product,
            "orders" => $count_order,
            "carts" => $count_cart,
            "customers" => $count_customer,
            'specificprices' => $count_specificprice,
            'countries' => $count_country,
        );
        $idsToTreat = array(
            "products" => $last_id_product,
            "orders" => $last_id_order,
            "carts" => $last_id_cart,
            "customers" => $last_id_customer,
            "specificprices" => $last_id_specificprice,
            "countries" => $last_id_country
        );
        $string = 'TREATED customers: '.$count_customer.'/'.$total_customers.', orders: '.$count_order.'/'.$total_orders.', products: '.$count_product.'/'.$total_products.', carts: '.$count_cart.'/'.$total_carts.', specificprices: '.$count_specificprice.'/'.$total_specificprices.', countries: '.$count_country.'/'.$total_countries;
        $this->toCaller->sendRequest($string);
        $logs[] = $string;

        ksort($idsTreated);
        ksort($idsToTreat);
        ksort($totalsToTreat);
        ksort($treated);
        ksort($alreadyTreated);
        $reportData = array(
            'fullyCompleted' => false,
            'logs' => $logs,
            'time' => ($end - $begin),
            'memory' => memory_get_peak_usage(),
            'processValues' => array(
                'do_cart' => $do_cart,
                'do_customer' => $do_customer,
                'do_product' => $do_product,
                'do_order' => $do_cart,
                'do_specificprice' => $do_specificprice,
                'do_country' => $do_country,
                'idsAlreadyTreated' => $idsTreated,
                'idsTreated' => $idsToTreat,
                'totalAlreadyTreated' => $alreadyTreated,
                'totalTreated' => $treated,
                'totalToTreat' => $totalsToTreat,
            ),
        );

        if ($errorsInRow > 0) {
            return array(
                'error' => 'Erreur lors de la synchronisation',
                'errorCode' => 'error_sync',
                'success' => false,
                'payload' => $reportData
            );
        }

        if ($do_cart === false && $do_customer === false && $do_product === false && $do_order === false && $do_specificprice === false && $do_country === false) {
            $reportData['fullyCompleted'] = true;

            $this->toCaller->stopSync();
            Configuration::updateValue('DONE_SYNCHRO', "1", null, null, $shop->id);

            return array(
                'error' => null,
                'errorCode' => null,
                'success' => true,
                'payload' => $reportData,
            );
        }

        return array(
            'error' => null,
            'errorCode' => null,
            'success' => true,
            'payload' => $reportData
        );

        // $categories = $this->getCategories();
        // $customer_groups = $this->getDB('customer_group');
        // $specific_price_rules = $this->getDB('specific_price_rule');
        // $groups = $this->getDB('group');
        // $group_reductions = $this->getDB('group_reduction');
        // $specific_prices = $this->getDB('specific_price');
        // $group_langs = $this->getDB('group_lang');
    }


    public function isSyncActive($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $active_synchro = Configuration::get('ACTIVE_SYNCHRO', false, null, $shop->id);
        if ($active_synchro == "1") {
            return true;
        } else {
            return false;
        }
    }

    public function hookDisplayHeader($params)
    {
        try {
            if (!Tools::getIsset('toatlg')) {
                return;
            }

            if (!$this->canExecuteHook('header')) {
                return;
            }

            $token = Tools::getValue('toatlg');
            $customer = KilibaAutolog::findCustomerByToken($token);
            if ($customer === false) {
                Tools::redirect(Context::getContext()->link->getPageLink('index'));
                exit;
            }

            $id_cart = Tools::getValue('toatlgcart', null);
            KilibaAutolog::login($customer, $id_cart);

            $back = Tools::getValue('toatlgback');
            if (!empty($back)) {
                $back = urldecode($back);
                Tools::redirect('location:' . $back);
                exit;
            }

            Tools::redirect('location:'.Context::getContext()->link->getPageLink('index'));
            exit;
        } catch (Exception $e) {
            $toLog = new KilibaLog();
            $toLog->process = 'Autolog';
            $toLog->message = json_encode(array(
                'error' => $e->getMessage()
            ));
            $toLog->type = KilibaLog::LOG_TYPE_ERROR;
            $toLog->save();
        }
    }

    // USE HOOK OF FOOTER TO SEND DATA RELATED TO VISITS OF PRODUCTS OR CATEGORY FROM LOGGED CUSTOMERS ( OR UNLOGGED CUSTOMERS WITH COOKIE )
    public function hookDisplayFooter($params)
    {
        try {
            if (!$this->canExecuteHook('footer')) {
                return;
            }

            $id_customer = (string)Context::getContext()->customer->id;
            if (empty($id_customer)) {
                $id_customer = (string)Context::getContext()->cookie->id_prestashop;
            }

            if (empty($id_customer)) {
                return;
            }

            if ($id_customer != Context::getContext()->cookie->id_prestashop) {
                Context::getContext()->cookie->__set('id_prestashop', (string)$id_customer);
            }

            $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, Context::getContext()->shop->id);

            if (empty($id_account)) {
                return;
            }

            if ((string)Context::getContext()->controller->php_self == "product") {
                if (version_compare(_PS_VERSION_, '1.5.6', '<')) {
                    $product_id = (string)Tools::getValue('id_product');
                } else {
                    $product_id = (string)Context::getContext()->controller->getProduct()->id;
                }
            } else {
                $product_id = '0';
            }

            if ((string)Context::getContext()->controller->php_self == "category") {
                if (version_compare(_PS_VERSION_, '1.5.6', '<')) {
                    $category_id = (string)Tools::getValue('id_category');
                } else {
                    $category_id = (string)Context::getContext()->controller->getCategory()->id;
                }
            } else {
                $category_id = '0';
            }

            $module_url = Tools::getProtocol(Tools::usingSecureMode()) . $_SERVER['REQUEST_URI'];
            $today = date('Y-m-d H:i:s');
            $topic = 'visits';
            $records = array(array('value' => array('url' => (string)$module_url, 'id_customer' => (string)$id_customer, 'date' => (string)$today, 'id_product' => (string)$product_id, 'id_category' => (string)$category_id), 'key' => array('id' => $id_account)));
            if ($product_id === '0' && $category_id === '0') {
                // DO NOTHING
            } else {
                $this->toCaller->postKafka($topic, $this->schemas['visit'], $this->schemas['key'], $records, true);
            }
        } catch (Exception $e) {
            $toLog = new KilibaLog();
            $toLog->process = 'Visits';
            $toLog->message = json_encode(array(
                'error' => $e->getMessage()
            ));
            $toLog->type = KilibaLog::LOG_TYPE_ERROR;
            $toLog->save();
        }
    }

    protected function canPrepare($object, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        if ($object->id_shop == $shop->id) {
            return true;
        }

        if ($object->id_shop_group != $shop->id_shop_group) {
            return false;
        }

        $class = get_class($object);

        $shopGroup = new ShopGroup($shop->id_shop_group);
        switch ($class) {
            case 'Customer':
                if ($shopGroup->share_customer) {
                    return true;
                }
                return false;
            case 'StockAvailable':
                if ($shopGroup->share_stock) {
                    return true;
                }
                return false;
            case 'Order':
            case 'Cart':
                if ($shopGroup->share_order) {
                    return true;
                }
                return false;
        }
    }

    protected function createHandleObject($params)
    {
        $object = $params['object'];
        $class = get_class($object);

        $share_type = false;
        switch ($class) {
            case 'Address':
            case 'Customer':
                $share_type = Shop::SHARE_CUSTOMER;
                break;
            case 'StockAvailable':
                $share_type = Shop::SHARE_STOCK;
                break;
            case 'Order':
                $share_type = Shop::SHARE_ORDER;
                break;
        }

        switch ($class) {
            case 'Product':
            case 'ProductAttribute':
                $shops = Shop::getShops(true, null, true);
                break;
            default:
                $shops = Shop::getContextListShopID($share_type);
                break;
        }

        $rand_key = Tools::passwdGen();

        self::keepCurrentContext($rand_key);
        foreach ($shops as $shop_id) {
            $shop = new Shop($shop_id);

            if (!self::isEnabled($shop)) {
                continue;
            }

            Shop::setContext(Shop::CONTEXT_SHOP, $shop->id);

            $this->handleObject($params, $shop);
        }
        self::restoreCurrentContext($rand_key);
    }

    protected static $context_saved = array();
    public static function keepCurrentContext($rand_key)
    {
        self::$context_saved[$rand_key] = array(
            'context' => Shop::getContext(),
            'context_shop' => Shop::getContextShopID(),
            'context_shop_group' => Shop::getContextShopGroupID(),
        );
    }

    public static function restoreCurrentContext($rand_key)
    {
        if (!isset(self::$context_saved[$rand_key])) {
            return;
        }

        $context = self::$context_saved[$rand_key];
        if ($context['context'] == Shop::CONTEXT_GROUP) {
            Shop::setContext(Shop::CONTEXT_GROUP, $context['context_shop_group']);
            return;
        }

        if ($context['context'] == Shop::CONTEXT_SHOP) {
            Shop::setContext(Shop::CONTEXT_SHOP, $context['context_shop']);
            return;
        }

        Shop::setContext(Shop::CONTEXT_ALL);
    }

    public function hookActionObjectAddAfter($params)
    {
        $this->createHandleObject($params);
    }

    // HOOK : CALL WHEN ADDING A PRODUCT OR MODIFYING A CART
    public function hookActionObjectUpdateAfter($params)
    {
        $this->createHandleObject($params);
    }

    public function hookActionObjectDeleteAfter($params)
    {
        $object = $params["object"];
        $class = get_class($object);

        foreach (Shop::getContextListShopID() as $shop_id) {
            $shop = new Shop($shop_id);
            $this->handleObject($params, $shop);

            switch ($class) {
                case "SpecificPrice":
                    if ($object->id_product != '0') {
                        $this->postProduct($object->id_product, null, false, true);
                    }
                    break;
                default:
                    // DO NOTHING
                    break;
            }
        }
    }

    public static $module_enabled = array();
    public static function isEnabled($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }
        if (!isset(self::$module_enabled[$shop->id])) {
            $id_module = Module::getModuleIdByName('kiliba');
            self::$module_enabled[$shop->id] = Db::getInstance()->getValue('SELECT `id_module` FROM `' . _DB_PREFIX_ . 'module_shop` WHERE `id_module` = ' . (int)$id_module . ' AND `id_shop` = ' . (int)$shop->id);
        }
        return self::$module_enabled[$shop->id];
    }

    public function canExecuteHook($hookType, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        if (!self::isEnabled($shop)) {
            return false;
        }

        if (Configuration::get('TO_HOOK_UPDATE_DISABLED', null, null, $shop->id)) {
            return false;
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        if (empty($id_account)) {
            return false;
        }

        if (Configuration::get('TO_HOOK_UPDATE_'.Tools::strtoupper($hookType).'_DISABLED', null, null, $shop->id)) {
            return false;
        }

        if (($hookType === "SpecificPrice" || $hookType === "Product" || $hookType === "ProductAttribute") &&
            Configuration::get('KILIBA_TO_GENERATE_MODE', null, null, $shop->id) === "xml") {
                // We do not execute the hook if the export mode for produits is by XML
                return false;
        }

        return true;
    }

    // HANDLE ADD/UPDATE OF OBJECT FROM PRESTASHOP HOOKS
    public function handleObject($params, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        try {
            $object = $params["object"];
            $class = get_class($object);
            // $this->sendRequest("ADD/UPDATE Object: ".$class." , ID: ".$object->id);
            if (!$this->canExecuteHook($class, $shop)) {
                return false;
            }

            if (!Validate::isLoadedObject($object)) {
                return false;
            }

            switch ($class) {
                case "Address":
                    $this->postCustomer($object->id_customer, $shop);
                    break;
                case "Customer":
                    $this->postCustomer($object->id, $shop);
                    break;
                case "ProductAttribute":
                    $this->postProduct($object->id_product, $shop);
                    break;
                case "Product":
                    // $this->sendRequest("$object");
                    // $keys = array_keys($params);
                    // $this->sendRequest(implode('||',$keys));
                    $this->postProduct($object->id, $shop);
                    break;
                case "Cart":
                    if (Configuration::get('TO_HOOK_UPDATE_DISABLED', null, null, $shop->id)) {
                        return false;
                    }
                    // DO NOTHING
                    if (!is_null($object)) {
                        $this->postCart($object, $shop, false);
                    }
                    break;
                case "Order":
                    $this->postOrder($object, $shop);
                    break;
                case "PrestaShopLogger":
                    // DO NOTHING
                    break;
                case "Condition":
                    // DO NOTHING
                    break;
                case "Combination":
                    // DO NOTHING
                    break;
                case "Connection":
                    // DO NOTHING
                    break;
                case "ProductSupplier":
                    // DO NOTHING
                    break;
                case "StockAvailable":
                    // DO NOTHING
                    break;
                case "ConnectionsSource":
                    // DO NOTHING
                    break;
                case "Guest":
                    // DO NOTHING
                    break;
                case "OrderDetail":
                    // DO NOTHING
                    break;
                case "OrderHistory":
                    // DO NOTHING
                    break;
                case "OrderCarrier":
                    // DO NOTHING
                    break;
                case "SpecificPriceRule":
                    // $this->sendRequest("SPECIFIC PRICE RULE: ".$object->id);
                    // DO NOTHING
                    break;
                case "SpecificPrice":
                    // TREAT REDUCTION
                    if ($object->id_product != '0') {
                        $this->postProduct($object->id_product);
                    }
                    // $this->sendRequest("ADD/UPDATE Object: ".$class." , ID REDUCTION: ".$object->id." , ID PRODUCT: ".$object->id_product);
                    // DO NOTHING
                    break;
                default:
                    // DO NOTHING
                    break;
            }
        } catch (Exception $e) {
            $toLog = new KilibaLog();
            $toLog->process = 'handleObject '.$class;
            $toLog->message = json_encode(array(
                'error' => $e->getMessage()
            ));
            $toLog->type = KilibaLog::LOG_TYPE_ERROR;
            $toLog->save();

            $this->toCaller->sendRequest('[ERROR] handleObject: '.$class.' , ID: '.$object->id.' : '.$e->getMessage());
        }
    }

    protected function getProductUrl($product, $shop = null, $relative_url = true)
    {
        if ($shop === null) {
            $shop = Context::getContext()->shop;
        }

        $url = Context::getContext()->link->getProductLink($product, null, null, null, null, $shop->id);
        $base_url = self::getBaseLink($shop);

        if ($relative_url === true) {
            return '/'.ltrim(str_replace($base_url, '', $url), '/');
        }

        return $url;
    }

    public static $cache = array();
    public static function getProductCategoryLink($product)
    {
        if (!isset(self::$cache[(int)$product->id]['category_url'])) {
            self::cacheProductInfos($product->id, array(
                'category_url' => Category::getLinkRewrite($product->id_category_default, Context::getContext()->language->id)
            ));
        }

        return self::$cache[(int)$product->id]['category_url'];
    }

    public static function cacheProductInfos($product_id, $infos = array())
    {
        $product_id = (int)$product_id;
        if (!isset(self::$cache[$product_id])) {
            self::$cache[$product_id] = array();
        }

        self::$cache[$product_id] = array_merge(self::$cache[$product_id], $infos);
    }

    public function tryPrepare($prepareType, $prepareId)
    {
        if ($prepareType == 'product') {
            return $this->prepareProduct($prepareId);
        }
        if ($prepareType == 'cart') {
            $cart = new Cart($prepareId);
            if (!Context::getContext()->cart) {
                Context::getContext()->cart = $cart;
            }
            if (!Context::getContext()->currency) {
                $currency = Currency::getCurrencyInstance(Configuration::get('PS_CURRENCY_DEFAULT', null, null, Context::getContext()->shop->id));
                Context::getContext()->currency = $currency;
            }

            return $this->prepareCart($cart);
        }
        if ($prepareType == 'order') {
            return $this->prepareOrder(new Order($prepareId));
        }
        if ($prepareType == 'customer') {
            return $this->prepareCustomer($prepareId);
        }
        if ($prepareType == 'specificPrice') {
            return $this->prepareSpecificPrice($prepareId);
        }

        return array();
    }

    protected function prepareSpecificPrice($specific_price_id, $shop = null)
    {
        if ($shop === null) {
            $shop = Context::getContext()->shop;
        }
        $id_shop = $shop->id;
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $id_shop);

        $specific_price = new SpecificPrice($specific_price_id);
        if (!Validate::isLoadedObject($specific_price)) {
            return array();
        }

        // Context not defined in PrestaShop 1.7 during the initial synchronisation with cron
        if (!Validate::isLoadedObject(Context::getContext()->currency) || empty(Context::getContext()->currency)) {
            Context::getContext()->currency = new Currency((int)Configuration::get('PS_CURRENCY_DEFAULT'));
        }

        $record = array();
        $record['id_specific_price'] = $specific_price->id;
        $record['id_specific_price_rule'] = $specific_price->id_specific_price_rule;
        $record['id_cart'] = $specific_price->id_cart;
        $record['id_product'] = $specific_price->id_product;
        $record['id_shop'] = $specific_price->id_shop;
        $record['id_shop_group'] = $specific_price->id_shop_group;
        $record['id_currency'] = $specific_price->id_currency;
        $record['id_country'] = $specific_price->id_country;
        $record['id_group'] = $specific_price->id_group;
        $record['id_customer'] = $specific_price->id_customer;
        $record['id_product_attribute'] = $specific_price->id_product_attribute;
        $record['price'] = $specific_price->price;
        $record['from_quantity'] = $specific_price->from_quantity;
        $record['reduction'] = $specific_price->reduction;
        $record['reduction_tax'] = (!isset($specific_price->reduction_tax) ? 0 : $specific_price->reduction_tax);
        $record['reduction_type'] = $specific_price->reduction_type;
        $record['from'] = $specific_price->from;
        $record['to'] = $specific_price->to;
        $records = array(array("value" => $record, "key" => array("id" => $id_account)));

        return $records;
    }

    // PREPARE PRODUCT RECORD FOR KAFKA
    public function prepareProduct($product_id, $shop = null, $isCron = false)
    {
        if ($shop === null) {
            $shop = Context::getContext()->shop;
        }
        $id_shop = $shop->id;
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $id_shop);

        $is_price_without_tax = (bool)Configuration::get('TO_PRICE_WITHOUT_TAX', null, null, $id_shop);

        /** @var Product $product */
        $product = new Product((int)$product_id, false, null, $id_shop);
        if (!Validate::isLoadedObject($product) || !$product->isAssociatedToShop($id_shop)) {
            return array();
        }

        if (Kiliba::urlProductHasCategory()) {
            if (empty($product->category)) {
                $product->category = Kiliba::getProductCategoryLink($product);
            }
        }

        $url_product = $this->getProductUrl($product, $shop);
        $url_product_absolute = $this->getProductUrl($product, $shop, false);

        if (!$is_price_without_tax) {
            /** @var TaxCalculator $tax_calculator */
            $tax_calculator = self::getTaxCalculator($product->id_tax_rules_group, 'FR');
        }
        $specificPricePriority = implode(";", SpecificPrice::getPriority($product->id));
        $base_price = $product->price;
        $name = array();
        $description = array();
        $wordings = $this->getWordingsOfProduct($product_id, $shop);
        foreach ($wordings as $wording) {
            $id_lang = (string)$this->getLang($wording["id_lang"]);
            $name[$id_lang] = $wording["name"];
            if (Tools::strlen($wording["name"]) === 0 ||  Tools::strlen($wording["name"]) === null) {
                $name[$id_lang] = "";
            } else {
                $name[$id_lang] = $wording["name"];
            }
            if (Tools::strlen($wording["description"]) === 0  || Tools::strlen($wording["description"]) === null) {
                $description[$id_lang] = "";
            } else {
                $description[$id_lang] = $wording["description"];
            }
        }
        if (count($name) === 0) {
            $name['fr'] = "";
        }
        if (count($description) === 0) {
            $description['fr'] = "";
        }
        $stock = $this->getStocksOfProduct($product_id, $shop);
        $attributes = $this->getAttributesOfProduct($product_id, $shop);

        $price = $base_price;
        $base_eco_tax = $product->ecotax;
        $ecotax = $product->ecotax;
        $ecotax_tax_calculator = self::getTaxCalculator((int)Configuration::get('PS_ECOTAX_TAX_RULES_GROUP_ID'));
        if (!$is_price_without_tax) {
            $price = $tax_calculator->addTaxes($base_price);
            $ecotax = $ecotax_tax_calculator->addTaxes($product->ecotax);
            $price += $ecotax;
        }
        $record = array();
        if ($isCron) {
            $record["id_product"] = (string)$product_id;
            $record["id_product_attribute"] = "0";
        } else {
            $record["id"] = (string)$product_id;
            $record["id_attribute"] = "0";
            $record["active"] = (string)$product->active;
            $record["available_for_order"] = (string)$product->available_for_order;
        }
        $record["id_shop_default"] = (string)$product->id_shop_default;
        $record["on_sale"] = (string)$product->on_sale;
        $record["id_category_default"] = (string)$product->id_category_default;
        $record["minimal_quantity"] = (string)$product->minimal_quantity;
        $record["visibility"] = (string)$product->visibility;
        $record["wholesale_price"] = (string)$product->wholesale_price;
        $record["priority"] = (string)$specificPricePriority;
        $record["price"] = (string)$price;
        $record["ecotax"] = (string)$ecotax;
        $record["available_when_out_of_stock"] = (string)Product::isAvailableWhenOutOfStock($stock['out_of_stock']);
        $record["date_add"] = (string)$product->date_add;
        $record["date_update"] = (string)$product->date_upd;
        $record["default_attribute"] = "0";
        if (empty($stock)) {
            $record["quantity"] = "0";
            $record["depends_on_stock"] = "0";
            $record["out_of_stock"] = "0";
            $record["reserved_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
            $record["physical_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
        } else {
            $record["quantity"] = (string)$stock["quantity"];
            if (version_compare(_PS_VERSION_, '1.4', '>=') && version_compare(_PS_VERSION_, '1.7', '<')) {
                // code spécifique aux versions 1.5.x & 1.6x
                $record["reserved_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
                $record["physical_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
            } else {
                $record["reserved_quantity"] = (string)$stock["reserved_quantity"]; // compatibility 1.5 & 1.6
                $record["physical_quantity"] = (string)$stock["physical_quantity"]; // compatibility 1.5 & 1.6
            }
            $record["depends_on_stock"] = (string)$stock["depends_on_stock"];
            $record["out_of_stock"] = (string)$stock["depends_on_stock"];
        }
        $record["image_url"] = (string)$this->getUrlImageOfProduct($product_id, false, $shop);
        $record["relative_url"] = (string)$url_product;
        $record["absolute_url"] = (string)$url_product_absolute;
        $record["name"] = $name;
        $record["description"] = $description;
        $record["reductions"] = array();

        if (empty($attributes)) {
            $records = array(array("key" => array("id" => $id_account), "value" => $record));
        } else {
            $records = array();
            $only_first_attribute = Configuration::get('TO_BATCH_PRODUCT_NO_ATTRIBUTES', null, null, $id_shop);

            $default_attribute = 0;
            foreach ($attributes as $attribute) {
                $price = $base_price + $attribute['price'];
                if ($attribute["ecotax"] == 0 || empty($attribute["ecotax"])) {
                    $ecotax = $product->ecotax;
                } else {
                    $ecotax = $attribute["ecotax"];
                }
                if (!$is_price_without_tax) {
                    $price = $tax_calculator->addTaxes($price);
                    $ecotax = $ecotax_tax_calculator->addTaxes($ecotax);
                    $price += $ecotax;
                }

                if (empty($default_attribute)) {
                    $default_attribute = $attribute['id_product_attribute'];
                }

                $stock = $this->getStocksOfAttribute($attribute['id_product'], $attribute['id_product_attribute'], $shop);
                if ($isCron) {
                    $record["id_product_attribute"] = (string)$attribute["id_product_attribute"];
                } else {
                    $record["id_attribute"] = (string)$attribute["id_product_attribute"];
                }

                $record["ecotax"] = (string)$ecotax;

                $record["price"] = (string)$price;
                $record["available_when_out_of_stock"] = (string)Product::isAvailableWhenOutOfStock($stock['out_of_stock']);
                $record["default_attribute"] = (string)$default_attribute;
                if (empty($stock)) {
                    $record["quantity"] = "0";
                    $record["depends_on_stock"] = "0";
                    $record["out_of_stock"] = "0";
                    $record["reserved_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
                    $record["physical_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
                } else {
                    $record["quantity"] = (string)$stock["quantity"];
                    if (version_compare(_PS_VERSION_, '1.4', '>=') && version_compare(_PS_VERSION_, '1.7', '<')) {
                        // code spécifique aux versions 1.5.x & 1.6x
                        $record["reserved_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
                        $record["physical_quantity"] = (string)"none"; // compatibility 1.5 & 1.6
                    } else {
                        $record["reserved_quantity"] = (string)$stock["reserved_quantity"]; // compatibility 1.5 & 1.6
                        $record["physical_quantity"] = (string)$stock["physical_quantity"]; // compatibility 1.5 & 1.6
                    }
                    $record["depends_on_stock"] = (string)$stock["depends_on_stock"];
                    $record["out_of_stock"] = (string)$stock["depends_on_stock"];
                }
                $record["image_url"] = (string) $this->getUrlImageOfProduct($product_id, $record['id_attribute'], $shop);

                array_push($records, array("key" => array("id" => $id_account), "value" => $record));
                if ($only_first_attribute) {
                    break;
                }
            }
        }

        return $records;
    }

    public static function urlProductHasCategory()
    {
        $product_rule = Configuration::get('PS_ROUTE_PRODUCT_RULE', null, null, Context::getContext()->shop->id);
        if ($product_rule) {
            if (strpos($product_rule, 'category') !== false || strpos($product_rule, 'categories') !== false) {
                return true;
            }

            return false;
        }

        $default_routes = Dispatcher::getInstance()->default_routes;
        if (!isset($default_routes['product_rule']['rule'])) {
            return false;
        }

        return (strpos($default_routes['product_rule']['rule'], 'category') !== false
            || strpos($default_routes['product_rule']['rule'], 'categories') !== false);
    }

    //POST PRODUCT TO KAFKA
    protected function postProduct($product_id, $shop = null, $sync = false, $isHook = false)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $topic = "products";
        $success = true;
        $records = $this->prepareProduct($product_id, $shop);
        if (empty($records)) {
            return;
        }

        $result = $this->toCaller->postKafka($topic, $this->schemas['product'], $this->schemas['key'], $records, $sync, true, $shop);
        if ($result == false) {
            $success = false;
        }
        return $success;
    }

    // PREPARE CUSTOMER RECORD FOR KAFKA
    protected function prepareCustomer($customer_id, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);

        $customer = new Customer($customer_id);
        if (!$this->canPrepare($customer, $shop)) {
            return false;
        }

        // Context not defined in PrestaShop 1.7 during the initial synchronisation with cron
        if (!Validate::isLoadedObject(Context::getContext()->currency) || empty(Context::getContext()->currency)) {
            Context::getContext()->currency = new Currency((int)Configuration::get('PS_CURRENCY_DEFAULT'));
        }

        $addresses = $this->getAddressesOfCustomer($customer->id);
        $formated_addresses = array();
        foreach ($addresses as $address) {
            $formated_address = array(
                'id_address' => (string)$address['id_address'],
                'id_country' => (string)$address['id_country'],
                'id_state' => (string)$address['id_state'],
                'address1' => (string)$address['address1'],
                'address2' => (string)$address['address2'],
                'postcode' => (string)$address['postcode'],
                'city' => (string)$address['city'],
                'phone' => (string)$address['phone'],
                'active' => (string)$address['active'],
                'deleted' => (string)$address['deleted'],
                'date_add' => (string)$address['date_add'],
                'date_update' => (string)$address['date_upd']
            );
            array_push($formated_addresses, $formated_address);
        }
        if (version_compare(_PS_VERSION_, '1.6', '<')) {
            $id_lang = "non renseigné"; // prob 1.5
        } else {
            $id_lang = $this->getLang($customer->id_lang);  //compatibility 1.5
        }

        $id_groups = $customer->getGroups();
        $record = array('value' => array(
            'email' => (string)$customer->email,
            'id_prestashop' => (string)$customer->id,
            'id_shop_group' => (string)$customer->id_shop_group,
            'id_shop' => (string)$customer->id_shop,
            'firstname' => (string)$customer->firstname,
            'lastname' => (string)$customer->lastname,
            'secure_key' => (string)$customer->secure_key,
            'birthday' => (string)$customer->birthday,
            'newsletter' => (string)$customer->newsletter,
            'id_gender' => (string)$customer->id_gender,
            'id_lang' => (string)$id_lang,
            'optin' => (string)$customer ->optin,
            'active' => (string)$customer->active,
            'date_add' => (string)$customer->date_add,
            'date_update' => (string)$customer->date_upd,
            'id_default_group' => (string)$customer->id_default_group,
            'id_groups' => (string)implode(',', $id_groups),
            'addresses' => $formated_addresses
        ), 'key' => array('id' => $id_account));
        return $record;
    }

    // POST CUSTOMER TO KAFKA
    protected function postCustomer($customer_id, $shop, $sync = false)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $topic = 'customers';
        $record = $this->prepareCustomer($customer_id, $shop);
        if (false === $record) {
            return true;
        }
        $result = $this->toCaller->postKafka($topic, $this->schemas['customer'], $this->schemas['key'], array($record), $sync, true, $shop);
        return $result;
    }

    // PREPARE CART RECORD FOR KAFKA
    protected function prepareCart($cart, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        if (!$id_account) {
            return false;
        }

        if (!Validate::isLoadedObject($cart)) {
            return false;
        }

        if (!$this->canPrepare($cart, $shop)) {
            return false;
        }

        $id_customer = (string)$cart->id_customer;
        if ($id_customer == "0") {
            return false;
        }

        // Context not defined in PrestaShop 1.7 during the initial synchronisation with cron
        if (!Validate::isLoadedObject(Context::getContext()->currency) || empty(Context::getContext()->currency)) {
            Context::getContext()->currency = new Currency((int)Configuration::get('PS_CURRENCY_DEFAULT'));
        }

        $products = array();
        try {
            $products = $cart->getProducts();
        } catch (Exception $e) {
            return false;
        }

        $productsInfoToPush = array();
        foreach ($products as $product) {
            if (version_compare(_PS_VERSION_, '1.7', '<')) {
                $reduction = "0";      //prob under 1.7
                $reduction_type = "0";
            } else {
                $reduction = $product['reduction'];           // compatibility
                $reduction_type = $product['reduction_type'];
            }
            $total_unit_wt = $product['total_wt']/$product['cart_quantity'];
            $temp_product= array(
                'id_product' => (string)$product['id_product'],
                'id_product_attribute' => (string)$product['id_product_attribute'],
                'cart_quantity' => (string)$product['cart_quantity'],
                'id_shop' => (string)$product['id_shop'],
                'reference' => (string)$product['reference'],
                'reduction' => (string)$reduction,
                'reduction_type' => (string)$reduction_type,
                'total_wt' => (string)$product['total_wt'],
                'total_unit_wt' => (string)$total_unit_wt,
                'id_category_default' => (string)$product['id_category_default']
            );
            array_push($productsInfoToPush, $temp_product);
        }
        $record = array("value" => array(
            "id" => (string)$cart->id,
            "id_shop_group" => (string)$cart->id_shop_group,
            "id_shop" => (string)$cart->id_shop,
            "id_customer" => (string)$cart->id_customer,
            "id_address_invoice" => (string)$cart->id_address_invoice,
            "id_address_delivery" => (string)$cart->id_address_delivery,
            "date_add" => (string)$cart->date_add,
            "date_update" => (string)$cart->date_upd,
            "id_currency" => (string)$cart->id_currency,
            "total_with_tax_with_shipping_with_discount" => (string)$cart->getOrderTotal(true),
            "total_with_tax_without_shipping_with_discount" => (string)$cart->getOrderTotal(true, Cart::BOTH_WITHOUT_SHIPPING),
            "total_discount_with_tax" => (string)$cart->getOrderTotal(true, CartCore::ONLY_DISCOUNTS),
            "total_products_with_tax" => (string)$cart->getOrderTotal(true, Cart::ONLY_PRODUCTS),
            "products" => $productsInfoToPush
        )
        , "key" => array("id" => $id_account)
        );

        return $record;
    }

    // POST CART TO KAFKA
    protected function postCart($cart, $shop = null, $sync = false)
    {
        if (!(Context::getContext()->controller instanceof FrontController)) {
            return;
        }

        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        // The cart is sometimes not correctly set when coming from an hook
        // We clear the cache and retrieve the cart again
        $newCart = new Cart();
        if (method_exists($newCart, 'clearCache')) {
            $newCart->clearCache(true);
        }
        $cart = new Cart($cart->id);

        if (!Context::getContext()->cart) {
            Context::getContext()->cart = $cart;
        }

        $topic = "carts";
        $record = $this->prepareCart($cart);
        $result = false;
        if ($record != false) {
            $result = $this->toCaller->postKafka($topic, $this->schemas['cart'], $this->schemas["key"], array($record), $sync, true, $shop);
        }
        return $result;
    }

    // PREPARE ORDER RECORD FOR KAFKA

    /**
     * @param Order $order
     */
    protected function prepareOrder($order, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        if (!Validate::isLoadedObject($order)) {
            return false;
        }

        if (!$this->canPrepare($order, $shop)) {
            return false;
        }

        // Context not defined in PrestaShop 1.7 during the initial synchronisation with cron
        if (!Validate::isLoadedObject(Context::getContext()->currency) || empty(Context::getContext()->currency)) {
            Context::getContext()->currency = new Currency((int)Configuration::get('PS_CURRENCY_DEFAULT'));
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        $products = $order->getProducts();
        $productsInfoToPush = array();
        foreach ($products as $product) {
            $cart_quantity = $product['product_quantity'];
            if (version_compare(_PS_VERSION_, '1.6', '<')) {
                $id_product_attribute = "unspecified";
            } else {
                $id_product_attribute = (string)$product["product_attribute_id"];
            }
            $temp_product= array(
                "id_product" => (string)$product["id_product"],
                "id_product_attribute" => (string)$id_product_attribute,
                "cart_quantity" => (string)$cart_quantity,
                "reference" => (string)$product["reference"],
                "reduction_amount" => (string)$product["reduction_amount"],
                "reduction_percent" => (string)$product["reduction_percent"],
                "group_reduction" => (string)$product["group_reduction"],
                "price" => (string)$product["price"],
                "total_wt" => (string)$product["total_wt"],
                "id_category_default" => (string)$product["id_category_default"]
            );
            array_push($productsInfoToPush, $temp_product);
        }

        // "total_paid" => (string)$order->getTotalPaid(),
        // "total_with_tax" => (string)$order->total_paid_tax_incl,
        // "total_without_tax" => (string)$order->total_paid_tax_excl,
        // "total_products_with_tax" => (string)$order->getTotalProductsWithTaxes(),
        // "total_products_without_tax" => (string)$order->getTotalProductsWithoutTaxes(),
        // "total_shipping_with_tax" => (string)$order->total_shipping_tax_incl,
        // "total_shipping_without_tax" => (string)$order->total_shipping_tax_excl,
        // "total_discount_with_tax" => (string)$order->total_discounts_tax_incl,
        // "total_discount_without_tax" => (string)$order->total_discounts_tax_excl,
        // temporary fix for huge orders
        $record = array("value" => array(
            "id" => (string)$order->id,
            "id_shop_group" => (string)$order->id_shop_group,
            "id_shop" => (string)$order->id_shop,
            "id_customer" => (string)$order->id_customer,
            "id_cart" => (string)$order->id_cart,
            "valid" => (string)$order->valid,
            "current_state" => (string)$order->current_state,
            "reference" => (string)$order->reference,
            "id_address_invoice" => (string)$order->id_address_invoice,
            "id_address_delivery" => (string)$order->id_address_delivery,
            "date_add" => (string)$order->date_add,
            "date_update" => (string)$order->date_upd,
            "id_currency" => (string)$order->id_currency,
            "total_paid" => "0",
            "total_with_tax" => "0",
            "total_without_tax" => "0",
            "total_products_with_tax" => "0",
            "total_products_without_tax" => "0",
            "total_shipping_with_tax" => "0",
            "total_shipping_without_tax" => "0",
            "total_discount_with_tax" => "0",
            "total_discount_without_tax" => "0",
            "products" => $productsInfoToPush
        ), "key" => array("id" => $id_account));
        return $record;
    }

    // POST ORDER TO KAFKA
    protected function postOrder($order, $shop = null, $sync = false)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $topic = "orders";
        $record = $this->prepareOrder($order, $shop);
        if (false === $record) {
            return true;
        }
        $result = $this->toCaller->postKafka($topic, $this->schemas['order'], $this->schemas["key"], array($record), $sync, true, $shop);
        return $result;
    }

    /**
     * CALLS IN THE DATABASE
     */
    protected function getUrlImageOfProduct($id_product, $id_product_attribute = false, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $images = $final_images = array();
        foreach (Db::getInstance()->executeS('
            SELECT i.`id_image`, i.`id_product`, i.`position`, `is`.`cover`
            FROM `' . _DB_PREFIX_ . 'image` i
            INNER JOIN `' . _DB_PREFIX_ . 'image_shop` `is` ON `i`.`id_image` = `is`.`id_image`
            WHERE `i`.`id_product` = ' . (int)$id_product . '
             AND `is`.`id_shop` = '.(int)$shop->id) as $image) {
            $images[$image['id_image']] = $image;
        }

        if ($id_product_attribute != false) {
            $attribute_images = Db::getInstance()->executeS('
           SELECT *
           FROM `' . _DB_PREFIX_ . 'product_attribute_image`
           WHERE `id_product_attribute` = ' . (int)$id_product_attribute . ' ');
            foreach ($attribute_images as $attribute_image) {
                if (empty($attribute_image['id_image'])) {
                    continue;
                }

                if (!isset($images[$attribute_image['id_image']])) {
                    continue;
                }

                $final_images[] = $images[$attribute_image['id_image']];
            }

            if (!empty($final_images)) {
                $images = $final_images;
            }
        }

        if (empty($images)) {
            return 'NONE';
        }

        $main_image = null;
        $position = 10000;
        foreach ($images as $image) {
            if ($image['cover'] == "1") {
                $main_image = $image;
                break;
            }

            if ($image['position'] <= $position) {
                $position = $image['position'];
                $main_image = $image;
            }
        }
        if ($main_image == null) {
            return 'NONE';
        }

        $imagePath = self::getBaseLink($shop).'img/p/'.Image::getImgFolderStatic($main_image['id_image']);
        $imagePath = $imagePath.(string)$main_image['id_image'].'.jpg';
        return $imagePath;
    }

    // LIST OF THE PRODUCTS OF A PRODUCT
    protected function getAttributesOfProduct($id_product, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $result = Db::getInstance()->executeS('
            SELECT `pa`.*, `pas`.*
            FROM `'._DB_PREFIX_.'product_attribute` `pa` INNER JOIN `'. _DB_PREFIX_ .'product_attribute_shop` `pas` ON `pas`.`id_product_attribute` = `pa`.`id_product_attribute`
            WHERE `pa`.`id_product` = ' . (int)$id_product . '
            AND `pas`.`id_shop` = '.(int)$shop->id.'
            ORDER BY `pas`.`default_on` DESC');

        $final_result = array_filter($result, function ($product_attribute) {
            if (array_key_exists("default_on", $product_attribute)) {
                if ($product_attribute["default_on"] == 1) {
                    return true;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        });

        return $final_result;
    }

    // GET WORDINGS OF PRODUCT
    protected function getWordingsOfProduct($id_product, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $result = Db::getInstance()->executeS('
          SELECT id_lang, id_shop, name, description
          FROM `' . _DB_PREFIX_ . 'product_lang`
          WHERE `id_product` = ' . pSQL($id_product) . '
          AND id_shop = '.(int)$shop->id);
        return $result;
    }

    // LIST ADDRESSES OF A PRODUCT
    protected function getAddressesOfCustomer($id_customer)
    {
        $result = Db::getInstance()->executeS('
            SELECT *
            FROM `'._DB_PREFIX_.'address` WHERE `id_customer` = '. (int)$id_customer);
        return $result;
    }

    // RETRIEVE ALL TABLE DATA IN PRESTASHOP
    protected function getDB($name, $limit, $offset)
    {
        $result = Db::getInstance()->executeS('
             SELECT *
             FROM `'._DB_PREFIX_.bqSQL($name).'` LIMIT '.(int)$limit.' OFFSET '.(int)$offset);
        return $result;
    }

    // RETRIEVE ALL TABLE DATA IN PRESTASHOP
    protected function getDBById($name, $id, $limit, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_name = 'id_'.$name;
        $shop_method = 'inner';
        if ($name == 'orders') {
            $id_name = 'id_order';
        }
        if ($name == 'product') {
            $shop_method = 'outer';
        }

        $result = Db::getInstance()->executeS('
             SELECT `a`.`'.bqSQL($id_name).'`
             FROM `'._DB_PREFIX_.bqSQL($name).'` `a`
             '.($shop_method == 'outer' ? ' INNER JOIN `'._DB_PREFIX_.pSQL($name).'_shop` `as` ON `a`.`'.$id_name.'` = `as`.'.$id_name : '').'
             WHERE `a`.`'.bqSQL($id_name).'` > '.(int)$id.'
             AND '.($shop_method == 'outer' ? '`as`' : '`a`').'.id_shop = '.(int)$shop->id.'
             ORDER BY `a`.`'.bqSQL($id_name).'` ASC
             LIMIT '.(int)$limit.'');
        return $result;
    }

    // GET STOCKS OF A PRODUCT
    protected function getStocksOfProduct($id_product, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $result = Db::getInstance()->getRow('
             SELECT *  FROM `' . _DB_PREFIX_ . 'stock_available`
             WHERE `id_product` = ' . (int)$id_product . ' '
            . StockAvailable::addSqlShopRestriction(null, $shop));
        return $result;
    }

    //GET STOCKS OF AN ATTRIBUTE OF A PRODUCT
    protected function getStocksOfAttribute($id_product, $id_product_attribute, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $result = Db::getInstance()->getRow('SELECT *  FROM `' . _DB_PREFIX_ . 'stock_available`
            WHERE `id_product` = '. (int)$id_product . '
            AND `id_product_attribute` = ' . (int)$id_product_attribute.' '
            . StockAvailable::addSqlShopRestriction(null, $shop));

        return $result;
    }

    // // RETRIEVE CUSTOMER INFO
    // public function getCustomerInfo($id)
    // {
    //     $result = Db::getInstance()->getRow('
    //  SELECT *
    //  FROM `'._DB_PREFIX_.'customer` c
    //  WHERE c.`id_customer` = '.pSQL($id));
    //     if (isset($result['id_customer'])) {
    //         return $result;
    //     }
    //     return false;
    // }

    // // PRODUCT CATEGORIES
    // public function getCategories()
    // {
    //     $result = Db::getInstance()->executeS('
    //     SELECT *
    //     FROM `'._DB_PREFIX_.'category` c
    //     INNER JOIN `'._DB_PREFIX_.'category_lang` cl
    //     ON c.id_category = cl.id_category
    //     ORDER BY c.id_category ASC');
    //     return $result;
    // }

    // GET CART FROM ID
    protected function getCart($id_cart)
    {
        $result = Db::getInstance()->getRow('
        SELECT *
        FROM `'._DB_PREFIX_.'cart` c
        WHERE c.`id_cart` = ' . (int)$id_cart . '');
        if (isset($result['id_cart'])) {
            return $result;
        }
        return false;
    }

    // RETRIEVE 5 LAST ACTIVE PRODUCTS ADDED IN PRESTASHOP
    protected function getLastProducts($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $result = Db::getInstance()->executeS('
             SELECT `p`.`id_product`
             FROM `'._DB_PREFIX_.'product` `p`
             INNER JOIN `'._DB_PREFIX_.'product_shop` `ps` ON `p`.`id_product` = `ps`.`id_product`
             WHERE `ps`.`active` = 1
             AND `ps`.`id_shop` = '.(int)$shop->id.'
             ORDER BY `p`.`date_add` DESC
             LIMIT 0,5');
        return $result;
    }

    // GET ISO CODE FROM ID_LANG
    protected function getLang($id_lang)
    {
        return (string)Language::getIsoById($id_lang);
    }

    // GET TOTAL NUMBER OF LINES IN A TABLE
    public static function getCountOfTable($name, $id = null, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_name = 'id_'.$name;
        $shop_method = 'inner';
        if ($name == 'orders') {
            $id_name = 'id_order';
        }
        if ($name == 'product') {
            $shop_method = 'outer';
        }
        $where = ($shop_method == 'outer' ? '`as`' : '`a`').'.`id_shop` = '.(int)$shop->id;
        if ($name == 'specific_price') {
            $where .= '(`a.`id_shop` = '.(int)$shop->id.' OR `a`.`id_shop` = 0 OR `id_shop_group` = '.(int)$shop->id_shop_group.') AND (`a`.`date_to` >= "'.date('Y-m-d H:i:s')."' OR `a`.`date_to` = '0000-00-00 00:00:00')";
        }
        if ($name == 'country') {
            $where .= '1';
        }

        $result = Db::getInstance()->getValue('
           SELECT COUNT(*) as total
           FROM `'._DB_PREFIX_.bqSQL($name).'` `a`
           '.($shop_method == 'outer' ? ' INNER JOIN `'._DB_PREFIX_.bqSQL($name).'_shop` `as` ON `a`.`'.bqSQL($id_name).'` = `as`.'.bqSQL($id_name) : '').'
           WHERE '.$where.'
           '. (($id !== null && $id > 0) ? ' AND `a`.`'.$id_name.'` <= '.(int)$id : '').'
           ORDER BY  `a`.`'.bqSQL($id_name).'` ASC');

        return (int)$result;
    }

    // GET TOTAL NUMBER OF LINES IN A TABLE
    public static function getTotalNoOfTable($name, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_name = 'id_'.$name;
        $shop_method = 'inner';
        if ($name == 'orders') {
            $id_name = 'id_order';
        }
        if ($name == 'product') {
            $shop_method = 'outer';
        }
        $where = ($shop_method == 'outer' ? '`as`' : '`a`').'.`id_shop` = '.(int)$shop->id;
        if ($name == 'specific_price') {
            $where .= '(`a.`id_shop` = '.(int)$shop->id.' OR `a`.`id_shop` = 0 OR `id_shop_group` = '.(int)$shop->id_shop_group.') AND (`a`.`date_to` >= "'.date('Y-m-d H:i:s')."' OR `a`.`date_to` = '0000-00-00 00:00:00')";
        }
        if ($name == 'country') {
            $where .= '1';
        }

        $sql = 'SELECT COUNT(*) as total
        FROM `'._DB_PREFIX_.pSQL($name).'` `a`
        '.($shop_method == 'outer' ? ' INNER JOIN `'._DB_PREFIX_.bqSQL($name).'_shop` `as` ON `a`.`'.bqSQL($id_name).'` = `as`.'.bqSQL($id_name) : '').'
        WHERE '.$where;

        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($sql);
        return $result['total'];
    }

    public static $cache_country_by_iso = array();
    public static function getIdCountrByIso($iso_code)
    {
        if (!isset(self::$cache_country_by_iso[$iso_code])) {
            self::$cache_country_by_iso[$iso_code] = Country::getByIso('FR');
        }

        return self::$cache_country_by_iso[$iso_code];
    }

    public static $cache_calculator = array();
    public static function getTaxCalculator($id_tax_rules, $iso_code = 'FR')
    {
        $cache_key = $id_tax_rules.'-'.$iso_code;
        if (!isset(self::$cache_calculator[$cache_key])) {
            $address = new Address();
            $address->id_country = (int)self::getIdCountrByIso($iso_code);
            $address->id_state = 0;
            $address->postcode = 0;

            $tax_manager = TaxManagerFactory::getManager($address, $id_tax_rules);
            self::$cache_calculator[$cache_key] = $tax_manager->getTaxCalculator();
        }

        return self::$cache_calculator[$cache_key];
    }
    // GET TAX FOR PRODUCT
    protected function getTax($id_tax_rules_group)
    {
        $tax_calculator = self::getTaxCalculator($id_tax_rules_group, 'FR');
        return $tax_calculator->getTotalRate();
    }

    // RETRIEVE ALL NO ORDER
    protected function getTotalNoOfOrder($id_customer)
    {
        $sql = 'SELECT count(`id_order`) as total FROM `' . _DB_PREFIX_ . 'orders`
        WHERE `id_customer` = ' . (int)$id_customer . ' AND `valid` = 1';
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($sql);
        return $result['total'];
    }
}
