<?php
/**
 *
 *  2018-2018 Kiliba
 *
 *  @author    Kiliba <support@kiliba.com>
 *  @copyright 2018 Kiliba
 *  @license Kiliba
 *
 *  AI in Digital Marketing
 *
 */

require_once(dirname(__FILE__).'/../../kiliba.php');
require_once(dirname(__FILE__).'/../../vendor/autoload.php');
require_once(dirname(__FILE__).'/../../classes/ProductEntry.php');

class KilibaXmlModuleFrontController extends ModuleFrontController
{
    /** @var int minimal_interval in second between difference call*/
    public $minimal_interval;

    /** @var date last call for curl  */
    public $last_call;

    /** @var int nb iterator xml */
    public $nb_iterator;

    /** @var date last generated xml  */
    public $last_completed;

    /** @var Collection products  */
    public $products;

    public $customer;

    /** @var Kiliba */
    public $module;
    protected $moduleName;

    public function __construct()
    {
        $this->context = Context::getContext();
        $this->context->controller = $this;
        $this->module = Module::getInstanceByName(Tools::getValue('module'));
        $this->moduleName = $this->module->name;
    }

    public function init()
    {
        if (!KilibaApi::checkToken(Tools::getValue('token'))) {
            KilibaApi::cutWithError('Token invalide');
        }

        if (!$this->isCurlInstalled(true)) {
            KilibaApi::cutWithError('Error curl not installed');
        }

        $this->customer = Configuration::get('ID_ACCOUNT_THATSOWL');
        $id_shop = $this->context->shop->id;
        if (!$this->canGenerate(Tools::getValue('current'))) {
            $minInterval = (int)Configuration::get('KILIBA_XML_MINIMAL_INTERVAL');
            $waitTime = $minInterval - (time() - strtotime(Configuration::get('KILIBA_XML_LAST_CALL')));

            $errorMessage = 'Another export is already in progress.'
                .' Please wait for '.$waitTime.' seconds before trying again.';
            $this->addLog($errorMessage, 'warning');
            die($errorMessage);
        }

        $today =  date('Y-m-d H:i:s');
        Configuration::updateValue('KILIBA_XML_LAST_CALL', $today, false, null, $id_shop);
        if (empty(Tools::getValue('current'))) {
            $file = fopen($this->getFeedName(), 'w+');
            fwrite($file, '<?xml version="1.0" encoding="utf-8"?><products version="'.$this->module->version.'">');
            fclose($file);
        }

        $this->generate(Tools::getValue('current'));
    }

    public function getProducts($id_lang, $limit_from)
    {
        $sql = 'SELECT p.`id_product`, pl.`name`, pl.`description`, product_shop.`price`
                FROM `'._DB_PREFIX_.'product` p
                '.Shop::addSqlAssociation('product', 'p').'
                LEFT JOIN `'._DB_PREFIX_.'product_lang` pl
                ON (p.`id_product` = pl.`id_product` '.Shop::addSqlRestrictionOnLang('pl').')
                WHERE pl.`id_lang` = '.(int)$id_lang.
                ' AND product_shop.`visibility` not IN ("none")'.
                ' AND product_shop.`available_for_order`= 1'.
                ' AND product_shop.`active` = 1';
        $sql .= ' ORDER BY p.`id_product` ASC';
        if ($limit_from !== false) {
            $sql .= ' LIMIT '.(int)$limit_from.', '.(int)Configuration::get('KILIBA_XML_NB_PRODUCT_ITERATOR');
        }
        return Db::getInstance()->executeS($sql);
    }

    public function generate($current = 0)
    {
        $id_lang = Context::getContext()->language->id;
        $products = $this->getProducts($id_lang, $current);
        if (empty($current)) {
            $this->addLog('START XML EXPORT FOR '.$this->customer.'_products.xml');
        }

        if (empty($products)) {
            $this->closeFeed();

            // Remove previous feed an place the newly generated one
            if (file_exists($this->getFeedName(false))) {
                unlink($this->getFeedName(false));
            }
            
            rename($this->getFeedName(), $this->getFeedName(false));
            // Notify end of cron execution
            $this->addLog('XML EXPORT COMPLETED: '.$this->customer.'_products.xml');
            // Empty last known url
            $date = date('Y-m-d H:i:s');
            Configuration::updateValue('KILIBA_LAST_UPD_FILE', $date);
        } else {
            $file = fopen($this->getFeedName(), 'a+');
            foreach ($products as $product) {
                try {
                    $str = $this->getBaseData($product);
                    fwrite($file, $str);
                } catch (Exception $exp) {
                    $this->addLog(
                        "Error exporting product ID ".$product['id_product'].": ".
                        $exp->getMessage()." - ".$exp->getTraceAsString()
                    );
                }
            }
            fclose($file);

            $protocol_link = (Configuration::get('PS_SSL_ENABLED')) ? 'https://' : 'http://';
            $next_uri = $protocol_link.Tools::getHttpHost().__PS_BASE_URI__;
            $next_uri .= 'index.php?fc=module&module=kiliba&controller=xml';
            $next_uri .= '&token='.Configuration::get('KILIBA_FLUX_TOKEN');
            $next_uri .= '&current='.($current + Configuration::get('KILIBA_XML_NB_PRODUCT_ITERATOR'));
            $this->addLog('CURL - next URL to call: '.$next_uri);

            // Disconnect DB to avoid reaching max connections
            DB::getInstance()->disconnect();

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $next_uri);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($curl, CURLOPT_HEADER, false);
            curl_setopt($curl, CURLOPT_POST, false);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 10);
            curl_setopt($curl, CURLOPT_TIMEOUT, 1);
            curl_exec($curl);
            curl_close($curl);
        }
        die();
    }

    /* Default data, in Product Class */
    private function getBaseData($product)
    {
        $productEntry = new ProductEntry();
        $productEntry->id = $product['id_product'];
        //https://sabre.io/xml/writing/
        $service = new Sabre\Xml\Service();

        $output = $service->write('product', $productEntry);
        //remove header xml
        $output = str_replace('<?xml version="1.0"?>', '', $output);
        return $output;
    }

    private function closeFeed()
    {
        $file = fopen($this->getFeedName(), 'a+');
        fwrite($file, '</products>');
    }


    private function addLog($message, $type = 'info')
    {
        $toLog = new KilibaLog();
        $toLog->process = 'Xml';
        $toLog->message = $message;
        $toLog->type = $type;
        $toLog->save();
    }

    private function canGenerate($iterator)
    {
        // Retrieve the minimal delay in seconds between two export
        $sec = (int)Configuration::get('KILIBA_XML_MINIMAL_INTERVAL');

        if (!empty($iterator)) {
            // This is an iteration of the XML, so we can continue the export
            return true;
        }

        if (strtotime(Configuration::get('KILIBA_XML_LAST_CALL')) + $sec <= time() ||
            strtotime(Configuration::get('KILIBA_XML_LAST_CALL')) == 0) {
            // The last export was more than X seconds ago, so we can start this new one
            return true;
        }

        // We consider that another cron is already in progress
        return false;
    }

    /**
     * Manage feed name depending on currency and lang
     */
    private function getFeedName($tmp_file = true)
    {
        if ($tmp_file) {
            return _PS_MODULE_DIR_.'/'.$this->moduleName.'/'.$this->customer.'_products_tmp.xml';
        } else {
            return _PS_MODULE_DIR_.'/'.$this->moduleName.'/'.$this->customer.'_products.xml';
        }
    }

    /**
     * Function to check if curl is installed
     */
    private function isCurlInstalled($returnBoolean = false)
    {
        if ($returnBoolean) {
            return in_array('curl', get_loaded_extensions());
        }
        if (in_array('curl', get_loaded_extensions())) {
            $response = $this->l('Active (correct)');
        } else {
            $response = $this->l('Not installed (incorrect)');
        }
        return $response;
    }
}
