<?php
/**
 *
 *  2018-2018 Kiliba
 *
 *  @author    Kiliba <support@kiliba.com>
 *  @copyright 2018 Kiliba
 *  @license Kiliba
 *
 *  AI in Digital Marketing
 *
 */

require_once(dirname(__FILE__).'/../../kiliba.php');

class KilibaApiModuleFrontController extends ModuleFrontController
{
    protected $toApi;
    public $module;
    protected $moduleName;

    public function __construct()
    {
        $this->context = Context::getContext();
        $this->context->controller = $this;
        $this->toApi = new KilibaApi();
        $this->module = Module::getInstanceByName(Tools::getValue('module'));
        $this->moduleName = $this->module->name;

        if (version_compare(_PS_VERSION_, '1.5.2.0', '<=')) {
            $protocol_link = (Configuration::get('PS_SSL_ENABLED') ||
                Tools::usingSecureMode()) ? 'https://' : 'http://';
            $useSSL = ((isset($this->ssl) && $this->ssl && Configuration::get('PS_SSL_ENABLED')) ||
                Tools::usingSecureMode()) ? true : false;
            $protocol_content = ($useSSL) ? 'https://' : 'http://';
            $link = new Link($protocol_link, $protocol_content);
            Context::getContext()->link = $link;
        }
    }

    public function init()
    {
        if (!KilibaApi::checkToken(Tools::getValue('token'))) {
            KilibaApi::cutWithError('Token invalide');
        }

        $action = Tools::getValue('action');

        switch ($action) {
            case 'getLog':
                $this->actionGetLogs();
                break;
            case 'deleteLog':
                $this->actionDeleteLogs();
                break;
            case 'setConfiguration':
                $this->actionSetConfiguration();
                break;
            case 'debug':
                $this->actionDebug();
                break;
            case 'hook':
                $this->actionHook();
                break;
            case 'try':
                $this->actionTry();
                break;
            case 'execution':
                $this->actionExecution();
                break;
            case 'activation':
                $this->actionActivation();
                break;
            case 'process':
                $this->actionProcess();
                break;
        }

        KilibaApi::cutWithError('aucune action sélectionnée, merci d\'en sélectionner une');
    }

    public function actionProcess()
    {
        $process = Tools::getValue('process');
        switch ($process) {
            case 'specificPrice':
                $this->actionProcessSpecificPrice();
                // no break
            case 'countries':
                $this->actionProcessCountries();
        }

        KilibaApi::cutWithError('process non valide, merci d\'en sélectionner un');
    }

    public function actionProcessSpecificPrice()
    {
        $querySpecificPrices = new KilibaQuerySpecificPrices();
        if (Tools::getIsset('id')) {
            $querySpecificPrices->setLastId((int) Tools::getValue('id'));
        }

        $querySpecificPrices->execute();

        $reportData = array();
        $reportData['lastId'] = $querySpecificPrices->getLastId();
        $reportData['nextPage'] = null;

        if ($querySpecificPrices->hasError()) {
            KilibaApi::cutWithError('Erreur lors de la synchronisation', 'error_sync', $reportData);
        }

        if ($querySpecificPrices->hasNextResults()) {
            $reportData = array(
                'nextPage' => $this->toApi->getApiUrl(
                    null,
                    array(
                        'action' => 'process',
                        'process' => 'specificPrice',
                        'id' => $querySpecificPrices->getLastId()
                    )
                ),
            );
        }
        $this->toApi->cutWithSuccess($reportData);
    }

    public function actionProcessCountries()
    {
        $queryCountries = new KilibaQueryCountries();
        if (Tools::getIsset('id')) {
            $queryCountries->setLastId((int) Tools::getValue('id'));
        }

        $queryCountries->execute();

        $reportData = array();
        $reportData['lastId'] = $queryCountries->getLastId();
        $reportData['nextPage'] = null;

        if ($queryCountries->hasError()) {
            KilibaApi::cutWithError('Erreur lors de la synchronisation', 'error_sync', $reportData);
        }

        if ($queryCountries->hasNextResults()) {
            $reportData = array(
                'nextPage' => $this->toApi->getApiUrl(
                    null,
                    array('action' => 'process', 'process' => 'countries', 'id' => $queryCountries->getLastId())
                ),
            );
        }
        $this->toApi->cutWithSuccess($reportData);
    }

    public function actionGetLogs()
    {
        $this->toApi->getLogs(Tools::getValue('format'));
    }

    public function actionDeleteLogs()
    {
        $date = Tools::getValue('from_date', false);
        if (!$date) {
            KilibaApi::cutWithError('Vous devez saisir une date dans le paramètre from_date');
        }

        $date = new DateTime($date);
        if (!$date) {
            KilibaApi::cutWithError(
                'La date fournie en paramètre est invalide, Merci de saisir une date sous la forme YYYY-MM-DD HH:II:SS'
            );
        }

        $this->toApi->deleteLogs($date);
    }

    public function actionSetConfiguration()
    {
        $authorizedConfiguration = array(
            'TO_DEBUG_MOD',
            'DONE_SYNCHRO',
            'ACTIVE_SYNCHRO',
            'PAGES_PRODUCT',
            'PAGES_ORDER',
            'PAGES_CART',
            'PAGES_CUSTOMER',
            'TO_HOOK_UPDATE_DISABLED',
            'TO_HOOK_UPDATE_PRODUCT_DISABLED',
            'TO_HOOK_UPDATE_ORDER_DISABLED',
            'TO_HOOK_UPDATE_CART_DISABLED',
            'TO_HOOK_UPDATE_CUSTOMER_DISABLED',
            'TO_HOOK_UPDATE_ADDRESS_DISABLED',
            'TO_HOOK_UPDATE_SPECIFICPRICE_DISABLED',
            'TO_DURATION_SYNCH_INIT',
            'TO_DURATION_SYNCH',
            'TO_BATCH_PRODUCT_NO_ATTRIBUTES',
            'TO_TOKEN_SYNC',
            'TO_BATCH_SIZE_PRODUCT',
            'TO_BATCH_SIZE_GLOBAL',
            'TO_BATCH_SIZE_SPECIFICPRICE',
            'TO_BATCH_PRODUCT_DO',
            'TO_BATCH_CUSTOMER_DO',
            'TO_BATCH_ORDER_DO',
            'TO_BATCH_CART_DO',
            'TO_BATCH_SPECIFICPRICE_DO',
            'TO_BATCH_COUNTRIES_DO',
            'TO_PRICE_WITHOUT_TAX',
            'TO_SPECIFIC_PRICE_LIMIT',
            'TO_CURL_TIMEOUT',
            'TO_CURL_CONNECTTIMEOUT_MS',
            'TO_CURL_CONNECTTIMEOUT_MS_VISITS',
            'KILIBA_XML_NB_PRODUCT_ITERATOR',
            'KILIBA_TO_GENERATE_MODE',
            'KILIBA_XML_MINIMAL_INTERVAL'
        );

        $configuration = Tools::strtoupper((string)Tools::getValue('configuration'));
        if (!in_array($configuration, $authorizedConfiguration)) {
            KilibaApi::cutWithError(
                'Vous devez saisir une configuration valide : '.implode(',', $authorizedConfiguration)
            );
        }

        $value = (string)Tools::getValue('value');
        if (!$value && $value != "0") {
            $error = '';
            if ($configuration == 'TO_DEBUG_MOD') {
                $error = 'Vous devez saisir un level dans le paramètre level (info, warning, error)';
            }

            KilibaApi::cutWithError(
                'Vous devez saisir une valeur dans le paramètre value'.(!empty($error) ? ' - '.$error : '')
            );
        }

        if ($configuration === "KILIBA_TO_GENERATE_MODE" && ($value !== "hook" && $value !== "xml")) {
            KilibaApi::cutWithError(
                'Vous devez saisir un mode de generaiton valide : (hook, xml)'
            );
        }

        Configuration::updateValue($configuration, $value, false, null, Context::getContext()->shop->id);
        $this->toApi->cutWithSuccess(array(
            $configuration => $value
        ));
    }

    public function actionDebug()
    {
        $authorizedConfiguration = array(
            'TO_DEBUG_MOD',
            'DONE_SYNCHRO',
            'ACTIVE_SYNCHRO',
            'PAGES_PRODUCT',
            'PAGES_ORDER',
            'PAGES_CART',
            'PAGES_CUSTOMER',
            'TO_HOOK_UPDATE_DISABLED',
            'TO_HOOK_UPDATE_PRODUCT_DISABLED',
            'TO_HOOK_UPDATE_ORDER_DISABLED',
            'TO_HOOK_UPDATE_CART_DISABLED',
            'TO_HOOK_UPDATE_CUSTOMER_DISABLED',
            'TO_HOOK_UPDATE_ADDRESS_DISABLED',
            'TO_HOOK_UPDATE_SPECIFICPRICE_DISABLED',
            'TO_DURATION_SYNCH_INIT',
            'TO_DURATION_SYNCH',
            'TO_BATCH_PRODUCT_NO_ATTRIBUTES',
            'TO_TOKEN_SYNC',
            'PS_REWRITING_SETTINGS',
            'PS_ROUTE_PRODUCT_RULE',
            'TO_BATCH_SIZE_PRODUCT',
            'TO_BATCH_SIZE_GLOBAL',
            'TO_BATCH_SIZE_SPECIFICPRICE',
            'TO_BATCH_PRODUCT_DO',
            'TO_BATCH_CUSTOMER_DO',
            'TO_BATCH_ORDER_DO',
            'TO_BATCH_CART_DO',
            'TO_BATCH_SPECIFICPRICE_DO',
            'TO_BATCH_COUNTRIES_DO',
            'TO_SPECIFIC_PRICE_LIMIT',
            'PS_SSL_ENABLED',
            'PS_SSL_ENABLED_EVERYWHERE',
            'TO_PRICE_WITHOUT_TAX',
            'PS_STOCK_MANAGEMENT',
            'PS_ORDER_OUT_OF_STOCK',
            'TO_CURL_TIMEOUT',
            'TO_CURL_CONNECTTIMEOUT_MS',
            'TO_CURL_CONNECTTIMEOUT_MS_VISITS',
            'KILIBA_TO_GENERATE_MODE',
            'KILIBA_XML_MINIMAL_INTERVAL',
            'KILIBA_XML_LAST_CALL',
            'KILIBA_XML_NB_PRODUCT_ITERATOR',
            'KILIBA_LAST_UPD_FILE',
        );

        $kiliba = new Kiliba();

        $payload = array(
            'version' => $kiliba->version,
            'prestashopVersion' => _PS_VERSION_,
            'configs' => array(
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'ssl_enabled' => (Configuration::get('PS_SSL_ENABLED', null, null, Context::getContext()->shop->id)
                    && Configuration::get('PS_SSL_ENABLED_EVERYWHERE', null, null, Context::getContext()->shop->id))
            ),
            'vars' => array(
                'hooksEnabled' => !Configuration::get(
                    'TO_HOOK_UPDATE_DISABLED',
                    null,
                    null,
                    Context::getContext()->shop->id
                ),
                'moduleEnabled' => Module::isEnabled($this->moduleName),
                'urlProductHasCategory' => Kiliba::urlProductHasCategory(),
            ),
            'configurations' => array(),
            'hooks' => array(),
            'pagesUrl' => array(
                'order' => Context::getContext()->link->getPageLink('order')
            ),
            'shops' => array()
        );
        foreach ($authorizedConfiguration as $configuration) {
            $payload['configurations'][$configuration] = Configuration::get(
                $configuration,
                null,
                null,
                Context::getContext()->shop->id
            );
            $payload['configurations-general'][$configuration] = Configuration::get($configuration, null);
            // Configuration de Florent
            // $payload['configurations'][$configuration] =
            // Configuration::get($configuration, null, null, Context::getContext()->shop->id);
        }
        foreach (Shop::getShops(false) as $shop) {
            $shop['id_account_thatsowl'] = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop['id_shop']);
            $payload['shops'][] = $shop;
        }

        $hooks = $this->getHooks();
        $payload['hooks'] = $hooks;
        $this->toApi->cutWithSuccess($payload);
    }

    public function actionHook()
    {
        $action = Tools::getValue('hookAction');
        if (!in_array($action, array( 'register', 'unregister' ))) {
            KilibaApi::cutWithError(
                'Vous devez saisir une action valide : '.implode(',', array( 'register', 'unregister' ))
            );
        }
        $hookName = Tools::getValue('hookName');
        $hook = Hook::getIdByName($hookName);
        if (!$hook) {
            KilibaApi::cutWithError('Nom du hook invalide');
        }
        try {
            $kiliba = new Kiliba();
            if ($action == 'register') {
                $kiliba->registerHook($hookName);
            } elseif ($action == 'unregister') {
                $kiliba->unregisterHook($hookName);
            }
        } catch (Exception $e) {
            KilibaApi::cutWithError($e->getMessage());
        }
        $this->toApi->cutWithSuccess(array(
            'hooks' => $this->getHooks()
        ));
    }

    public function actionTry()
    {
        $kiliba = new Kiliba();
        $result = $kiliba->tryPrepare(Tools::getValue('tryType'), Tools::getValue('tryId'));
        $this->toApi->cutWithSuccess(array(
            'result' => $result
        ));
    }

    public function actionExecution()
    {
        $newStatus = Tools::getValue('status');
        if (empty($newStatus)|| !in_array($newStatus, array('enable', 'disable'))) {
            KilibaApi::cutWithError('Vous devez saisir un status valide : '.implode(',', array( 'enable', 'disable' )));
        }

        $isExecuted = !Configuration::get('TO_HOOK_UPDATE_DISABLED', null, null, Context::getContext()->shop->id);
        if ($newStatus == 'enable') {
            if ($isExecuted) {
                $this->toApi->cutWithSuccess(
                    array('message' => 'Les hooks sont déjà activés, aucune action réalisée')
                );
            }

            Configuration::updateValue('TO_HOOK_UPDATE_DISABLED', 0, false, null, Context::getContext()->shop->id);
            $this->toApi->cutWithSuccess(array( 'message' => 'Hooks activés avec succès' ));
        }

        if ($newStatus == 'disable') {
            if (!$isExecuted) {
                $this->toApi->cutWithSuccess(
                    array('message' => 'Les hooks sont déjà désactivés, aucune action réalisée')
                );
            }

            Configuration::updateValue('TO_HOOK_UPDATE_DISABLED', 1, false, null, Context::getContext()->shop->id);
            $this->toApi->cutWithSuccess(array( 'message' => 'Hooks désactivés avec succès' ));
        }

        KilibaApi::cutWithError(array( 'message' => 'Aucune action executé' ));
    }

    public function actionActivation()
    {
        $newStatus = Tools::getValue('status');
        if (empty($newStatus)|| !in_array($newStatus, array('enable', 'disable'))) {
            KilibaApi::cutWithError('Vous devez saisir un status valide : '.implode(',', array( 'enable', 'disable' )));
        }

        if (!Module::isInstalled('kiliba')) {
            KilibaApi::cutWithError('Le module n\'est pas installé : '.implode(',', array( 'enable', 'disable' )));
        }

        if ($newStatus == 'enable') {
            if (Module::isEnabled($this->moduleName)) {
                $this->toApi->cutWithSuccess(array( 'message' => 'Le module est déjà activé, aucune action réalisée' ));
            }
            try {
                Module::enableByName($this->moduleName);
            } catch (\Exception $e) {
                KilibaApi::cutWithError('Une erreur est arrivée lors de l\'activation du module', '', array(
                    'message' => $e->getMessage(),
                ));
            }
            $this->toApi->cutWithSuccess(array( 'message' => 'Module activé avec succès' ));
        }

        if ($newStatus == 'disable') {
            if (!Module::isEnabled($this->moduleName)) {
                $this->toApi->cutWithSuccess(
                    array('message' => 'Le module est déjà désactivé, aucune action réalisée')
                );
            }
            try {
                Module::disableByName($this->moduleName);
            } catch (\Exception $e) {
                KilibaApi::cutWithError('Une erreur est arrivée lors de la désactivation du module', '', array(
                    'message' => $e->getMessage(),
                ));
            }
            $this->toApi->cutWithSuccess(array( 'message' => 'Module désactivé avec succès' ));
        }

        KilibaApi::cutWithError(array( 'message' => 'Aucune action executé' ));
    }

    public function getHooks($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $hooksQuery = Db::getInstance()->executeS('SELECT h.name
        FROM '._DB_PREFIX_.'hook_module hm
        LEFT JOIN '._DB_PREFIX_.'hook h ON hm.id_hook = h.id_hook
        LEFT JOIN '._DB_PREFIX_.'module m ON hm.id_module = m.id_module
        WHERE m.name = "kiliba"
        AND hm.id_shop = '.(int)$shop->id);
        $hooks = array();
        foreach ($hooksQuery as $hook) {
            $hooks[] = $hook['name'];
        }
        return $hooks;
    }
}
