<?php
/**
 *
 *  2018-2018 Kiliba
 *
 *  @author    Kiliba <support@kiliba.com>
 *  @copyright 2018 Kiliba
 *  @license Kiliba
 *
 *  AI in Digital Marketing
 *
 */
require_once(__DIR__ . '/../KilibaLog.php');

class KilibaCaller
{
    const ENDPOINT = 'https://app.thatsowl.com/external/';
    const ENDPOINTKAFKA = 'https://app.thatsowl.com/kafkarest/';

    public static $timeout = 3000;
    public static $connect_timeout = 500;

    public function __construct()
    {
        if ($value = Configuration::get('TO_CURL_TIMEOUT', null, null, Context::getContext()->shop->id)) {
            self::$timeout = $value;
        }

        if ($value = Configuration::get('TO_CURL_CONNECTTIMEOUT_MS', null, null, Context::getContext()->shop->id)) {
            self::$connect_timeout = $value;
        }
    }

    public function buildCurlQuery()
    {
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => self::$timeout,
            CURLOPT_CONNECTTIMEOUT_MS => self::$connect_timeout,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_HTTPHEADER => array(
                'Cache-Control: no-cache',
                'Content-Type: application/x-www-form-urlencoded',
            ),
        ));

        return $curl;
    }

    public function buildCurlQueryKafka()
    {
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, Context::getContext()->shop->id);
        $kafka_rest_credential = Configuration::get('KAFKA_REST_PASSWORD', null, null, Context::getContext()->shop->id);
        if (!$id_account) {
            return false;
        }

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/vnd.kafka.avro.v2+json',
        ));
        curl_setopt($curl, CURLOPT_TIMEOUT, self::$timeout);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT_MS, self::$connect_timeout);

        if (version_compare(PHP_VERSION, '5.5.0', '<')) {
            curl_setopt($curl, CURLOPT_USERPWD, $id_account.':'.$kafka_rest_credential);
        } else {
            curl_setopt($curl, CURLOPT_USERNAME, $id_account);
            curl_setopt($curl, CURLOPT_PASSWORD, $kafka_rest_credential);
        }

        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);

        return $curl;
    }

    public function checkQuery($processName, $curl)
    {
        $toLog = new KilibaLog();
        $toLog->process = 'Call CURL ' . $processName;
        $toLog->message = json_encode(array(
            'Query' => curl_getinfo($curl)
        ));
        $toLog->type = KilibaLog::LOG_TYPE_INFO;
        $toLog->save();


        $response = curl_exec($curl);
        $error = curl_error($curl);
        $errorNumber = curl_errno($curl);
        $infos = curl_getinfo($curl);
        $responseCode = $infos['http_code'];
        $url = $infos['url'];

        $responseDecoded = json_decode($response, true);

        $toLog = new KilibaLog();
        $toLog->process = 'result CURL '.$processName;
        $toLog->message = json_encode(array(
            'Infos' => $infos,
            'Erreur' => ($response === false ? $errorNumber.' - '.$error : 'Aucune'),
            'Response' => ($response === false ? '' : $responseDecoded)
        ));
        $toLog->type = ($response === false || $responseCode != '200' ? KilibaLog::LOG_TYPE_ERROR : KilibaLog::LOG_TYPE_INFO);
        $toLog->save();

        return array(
            'success' => $response !== false && $responseCode == 200,
            'httpCode' => $responseCode,
            'response' => $responseDecoded,
            'infos' => $infos,
            'error' => $error,
            'errorNumber' => $errorNumber,
            'url' => $url,
        );
    }

    /**
     * @return boolean
     * LOG SIMPLY PROGRESSION OF SYNC AND ERRORS TO EXTERNAL API
     */
    public function sendRequest($content)
    {
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, Context::getContext()->shop->id);

        if (is_array($content)) {
            $content = json_encode($content);
        }
        $message = urlencode($content);
        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'message='.$message.'&id_account='.$id_account);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'log');
        $response = $this->checkQuery(__FUNCTION__, $curl);

        return $response;
    }

    public function updateModule($plug_version, $shop = null)
    {
        if(null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop);

        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'plug_version='.$plug_version.'&id_account='.$id_account);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'updatemodule');
        $response = $this->checkQuery(__FUNCTION__, $curl);

        return $response;
    }

    // CHECK IF SYNC CAN BE DONE AT START OF SYNC
    public function checkBeforeStartSync($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        $urls = $this->getBasicUrl($shop);
        $shops = json_encode($urls);
        $good_shops = urlencode($shops);

        $is_multishop = count(Shop::getShops()) > 1;
        $url_logo = urlencode(_PS_BASE_URL_.__PS_BASE_URI__.'img/'.Configuration::get('PS_LOGO'));
        $http = "http";
        $https = "https";
        $httpss = "httpss";
        $https_url_logo = str_replace($http, $https, $url_logo);
        $clean_url_logo = str_replace($httpss, $https, $https_url_logo);

        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'id_account='.$id_account.'&url_logo='.$clean_url_logo.'&shops='.$good_shops.'&multishop='.(int)$is_multishop);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'startsync');

        $response = $this->checkQuery(__FUNCTION__, $curl);

        return $response;
    }


    // SEND URL TO CALL TO EXTERNAL API TO LAUNCH NEXT BATCH OF INSTALLATION SYNC
    public function callSync($tokenSync = null, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        if (is_null($tokenSync)) {
            $tokenSync = md5(rand());
            Configuration::updateValue('TO_TOKEN_SYNC', $tokenSync, false, null, $shop->id);
        }
        $cronInfos = Kiliba::getCronUrl($tokenSync, $shop);

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        $url_sync = urlencode($cronInfos['url']);
        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'url='.$url_sync.'&id_account='.$id_account);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'callsync');
        $response = $this->checkQuery(__FUNCTION__, $curl);

        return array(
            'response' => $response,
            'cronInfos' => $cronInfos
        );
    }

    /**
     * @return boolean
     * CHECK FORM
     */
    public function checkForm($id_to_check)
    {
        $id_account = $id_to_check;
        $urls = $this->getBasicUrl(Context::getContext()->shop);
        $shops = json_encode($urls);
        $good_shops = urlencode($shops);
        $token = Configuration::get('KILIBA_FLUX_TOKEN');
        $kiliba = new Kiliba();
        $version = (string)$kiliba->version;
        $selected_shop = strval(Context::getContext()->shop->id);
        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'id_account='.$id_account.'&token='.$token.'&shops='.$good_shops.'&selected='.$selected_shop.'&version='.$version);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'checkform');
        $response = $this->checkQuery(__FUNCTION__, $curl);

        return $response;
    }

    // ADVISE THATSOWL THAT THE INSTALLATION SYNC IS FINISHED
    public function stopSync()
    {
        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, Context::getContext()->shop->id);
        $curl = $this->buildCurlQuery();
        curl_setopt($curl, CURLOPT_POSTFIELDS, 'id_account='.$id_account);
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINT.'stopsync');
        $response = $this->checkQuery(__FUNCTION__, $curl);

        if (!$response['success']) {
            return false;
        }

        $real_result = $response['response']['result'];
        return $real_result;
    }

    // POST RECORD TO KAFKA
    public function postKafka($topic, $value_schema, $key_schema, $records = array(), $sync = false, $first_call = true, $shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $id_account = Configuration::get('ID_ACCOUNT_THATSOWL', null, null, $shop->id);
        $done_synchro = Configuration::get('DONE_SYNCHRO', null, null, $shop->id);

        if ($id_account == "0" || ($done_synchro == "0" && $sync == false)) {
            // Trafic durinng initial sync
            return false;
        }

        // UNE LISTE DE RECORDS VIDES N'ENGENDRE PAS UNE REQUETE VERS KAFKAREST
        if (count($records) === 0) {
            return true;
        }

        $fulltopicname = $id_account.'_'.$topic ;

        $curl = $this->buildCurlQueryKafka();
        if ($topic == 'visits'
            && $connect_timeout = Configuration::get('TO_CURL_CONNECTTIMEOUT_MS_VISITS', null, null, Context::getContext()->shop->id)) {
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT_MS, $connect_timeout);
        }

        $data = json_encode(array('value_schema' => $value_schema, 'key_schema' => $key_schema, 'records' => $records));
        if ($records != array()) {
            curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        }
        curl_setopt($curl, CURLOPT_URL, self::ENDPOINTKAFKA.'topics/'.$fulltopicname);
        $response = $this->checkQuery(__FUNCTION__, $curl);

        if ($response['httpCode'] != 200) {
            $message_string = "ERROR: ".$topic." ".$response['httpCode'];
            $this->sendRequest($message_string);
            $jsonToSend = array(
                'id_account' => $id_account,
                'topics' => $topic,
                'records'=> $records,
                'statuscode' => $response['httpCode'],
                'response' => $response
            );
            $this->sendRequest($jsonToSend);
        }

        if ($response['httpCode'] == '413'
            && $first_call === true) {
            $message_string = "Split query";
            $this->sendRequest($message_string);

            $perpage = 20;
            if ($topic == "product") {
                $perpage = 5;
            }
            $count = 0;
            $result = true;
            for ($count = 0; $count <= count($records); $count=$count+$perpage) {
                $new_records = array_slice($records, $count, $perpage);
                $result &= $this->postKafka($topic, $value_schema, $key_schema, $new_records, $sync, false, $shop);
            }

            return $result;
        }

        // LORS DE LA SYNCHRO LES ERREURS DE SCHEMA NE BLOQUENT PAS LES BATCHS SUIVANTS
        if ($sync === true && $response['httpCode'] == '422') {
            return true;
        }

        if (!$response['success']) {
            return false;
        }

        return true;
    }

    // GET BASIC URL OF SHOPS
    private function getBasicUrl($shop = null)
    {
        if (null === $shop) {
            $shop = Context::getContext()->shop;
        }

        $shop->setUrl();

        return array(array(
            'id_shop' => $shop->id,
            'physical_uri' => $shop->physical_uri,
            'virtual_uri' => $shop->virtual_uri,
            'domain' => $shop->domain,
            'domain_ssl' => $shop->domain_ssl,
        ));
    }
}
