<?php
/**
 * Creative Elements - Elementor based PageBuilder
 *
 * @author    WebshopWorks, Elementor
 * @copyright 2019-2020 WebshopWorks.com & Elementor.com
 * @license   https://www.gnu.org/licenses/gpl-3.0.html
 */

namespace CE;

defined('_PS_VERSION_') or die;

class Stylesheet
{
    private $rules = array();

    private $devices = array();

    private $raw = array();

    /**
     * @param array $rules
     *
     * @return string
     */
    public static function parseRules(array $rules)
    {
        $parsed_rules = '';

        foreach ($rules as $selector => $properties) {
            $selector_content = self::parseProperties($properties);

            if ($selector_content) {
                $parsed_rules .= $selector . '{' . $selector_content . '}';
            }
        }

        return $parsed_rules;
    }

    /**
     * @param array $properties
     *
     * @return string
     */
    public static function parseProperties(array $properties)
    {
        $parsed_properties = '';

        foreach ($properties as $property_key => $property_value) {
            if ('' !== $property_value) {
                $parsed_properties .= $property_key . ':' . $property_value . ';';
            }
        }

        return $parsed_properties;
    }

    /**
     * @param string $device_name
     * @param string $device_max_point
     *
     * @return $this
     */
    public function addDevice($device_name, $device_max_point)
    {
        $this->devices[$device_name] = $device_max_point;

        asort($this->devices);

        return $this;
    }

    /**
     * @param string $selector
     * @param array|string $rules
     * @param string $device
     *
     * @return $this
     */
    public function addRules($selector, $rules = null, $device = 'desktop')
    {
        if (null === $rules) {
            preg_match_all('/([^\s].+?(?=\{))\{((?s:.)+?(?=}))}/', $selector, $parsed_rules);

            foreach ($parsed_rules[1] as $index => $selector) {
                $this->addRules($selector, $parsed_rules[2][$index], $device);
            }

            return $this;
        }

        if (!isset($this->rules[$device][$selector])) {
            $this->rules[$device][$selector] = array();
        }

        if (is_string($rules)) {
            $rules = array_filter(explode(';', trim($rules)));

            $ordered_rules = array();

            foreach ($rules as $rule) {
                $property = explode(':', $rule, 2);

                $ordered_rules[trim($property[0])] = trim($property[1], ' ;');
            }

            $rules = $ordered_rules;
        }

        $this->rules[$device][$selector] = array_merge($this->rules[$device][$selector], $rules);

        return $this;
    }

    public function addRawCss($css, $device = '')
    {
        if (!isset($this->raw[$device])) {
            $this->raw[$device] = array();
        }

        $this->raw[$device][] = trim($css);

        return $this;
    }

    public function getRules($device = null, $selector = null, $property = null)
    {
        if (!$device) {
            return $this->rules;
        }

        if ($property) {
            return isset($this->rules[$device][$selector][$property]) ? $this->rules[$device][$selector][$property] : null;
        }

        if ($selector) {
            return isset($this->rules[$device][$selector]) ? $this->rules[$device][$selector] : null;
        }

        return isset($this->rules[$device]) ? $this->rules[$device] : null;
    }

    public function __toString()
    {
        $style_text = '';

        $devices = array_reverse($this->devices);

        $devices_names = array_keys($devices);

        array_unshift($devices_names, 'desktop');

        foreach ($devices_names as $device_name) {
            if (empty($this->rules[$device_name])) {
                continue;
            }

            $rules = $this->rules[$device_name];

            $device_text = self::parseRules($rules);

            if ($device_text && isset($this->devices[$device_name])) {
                $device_text = '@media(max-width: ' . $this->devices[$device_name] . 'px){' . $device_text . '}';
            }

            $style_text .= $device_text;
        }

        foreach ($this->raw as $device_name => $raw) {
            $raw = implode("\n", $raw);

            if ($raw && isset($this->devices[$device_name])) {
                $raw = '@media(max-width: ' . $this->devices[$device_name] . 'px){' . $raw . '}';
            }

            $style_text .= $raw;
        }

        return $style_text;
    }
}
